// Emulator einer F800 mit K-Line Interface
/* Wartet auf eine Request-Sequenz von einem Tester (z.B. Motoscan) und sendet diese unverndert weiter in den PC
 * Wartet dann auf die entsprechende Response vom PC und bermittelt diese unverndert zurck an den Tester
 * RB4, Pin13, Output, K-Line Transceiver Enable
 * RB6, Pin11, Input,  K-Line Receive
 * RC2, Pin14, Output, K-Line Transmit
 * UTX, Pin10, Output, Transmit zur Konsole
 * URX, Pin12, Input,  Receive von Konsole
 * RC5, Pin5,  Output, Display Enable
 * RC4, Pin6,  I/O,    Extra Interface (Bidirectional)
 * RC3, Pin7,  Input,  Raw Select Schalter
 * Takt = 12MHz : Externer Quarz
 * K-Line Receive luft auf 10400 baud
 * UART zum PC luft mit 19200 baud
 * R5 und R7 sollten unbestckt sein, R4 ist berbrckt
 */

//Configuration-Bits - erstellt ber "Set Configuration Bits" - "Generate Source Code"
// CONFIG1L
#pragma config CPUDIV = NOCLKDIV// CPU System Clock Selection bits (No CPU System Clock divide)
#pragma config USBDIV = OFF     // USB Clock Selection bit (USB clock comes directly from the OSC1/OSC2 oscillator block; no divide)
// CONFIG1H
#pragma config FOSC = XT        // Oscillator Selection bits (XT oscillator)
#pragma config PLLEN = OFF      // 4 X PLL Enable bit (PLL is under software control)
#pragma config PCLKEN = ON      // Primary Clock Enable bit (Primary clock enabled)
#pragma config FCMEN = OFF      // Fail-Safe Clock Monitor Enable (Fail-Safe Clock Monitor disabled)
#pragma config IESO = OFF       // Internal/External Oscillator Switchover bit (Oscillator Switchover mode disabled)
// CONFIG2L
#pragma config PWRTEN = ON      // Power-up Timer Enable bit (PWRT enabled)
#pragma config BOREN = OFF      // Brown-out Reset Enable bits (Brown-out Reset disabled in hardware and software)
#pragma config BORV = 19        // Brown-out Reset Voltage bits (VBOR set to 1.9 V nominal)
// CONFIG2H
#pragma config WDTEN = OFF      // Watchdog Timer Enable bit (WDT is controlled by SWDTEN bit of the WDTCON register)
#pragma config WDTPS = 32768    // Watchdog Timer Postscale Select bits (1:32768)
// CONFIG3H
#pragma config HFOFST = OFF     // HFINTOSC Fast Start-up bit (The system clock is held off until the HFINTOSC is stable.)
#pragma config MCLRE = OFF      // MCLR Pin Enable bit (RA3 input pin enabled; MCLR disabled)
// CONFIG4L
#pragma config STVREN = ON      // Stack Full/Underflow Reset Enable bit (Stack full/underflow will cause Reset)
#pragma config LVP = ON         // Single-Supply ICSP Enable bit (Single-Supply ICSP enabled)
#pragma config BBSIZ = OFF      // Boot Block Size Select bit (512W boot block size)
#pragma config XINST = OFF      // Extended Instruction Set Enable bit (Instruction set extension and Indexed Addressing mode disabled (Legacy mode))
// CONFIG5L
#pragma config CP0 = OFF        // Code Protection bit (Block 0 not code-protected)
#pragma config CP1 = OFF        // Code Protection bit (Block 1 not code-protected)
// CONFIG5H
#pragma config CPB = OFF        // Boot Block Code Protection bit (Boot block not code-protected)
#pragma config CPD = OFF        // Data EEPROM Code Protection bit (Data EEPROM not code-protected)
// CONFIG6L
#pragma config WRT0 = OFF       // Table Write Protection bit (Block 0 not write-protected)
#pragma config WRT1 = OFF       // Table Write Protection bit (Block 1 not write-protected)
// CONFIG6H
#pragma config WRTC = OFF       // Configuration Register Write Protection bit (Configuration registers not write-protected)
#pragma config WRTB = OFF       // Boot Block Write Protection bit (Boot block not write-protected)
#pragma config WRTD = OFF       // Data EEPROM Write Protection bit (Data EEPROM not write-protected)
// CONFIG7L
#pragma config EBTR0 = OFF      // Table Read Protection bit (Block 0 not protected from table reads executed in other blocks)
#pragma config EBTR1 = OFF      // Table Read Protection bit (Block 1 not protected from table reads executed in other blocks)
// CONFIG7H
#pragma config EBTRB = OFF      // Boot Block Table Read Protection bit (Boot block not protected from table reads executed in other blocks)

//Pin-Definitionen
//Pin-Outputs sollten ber das LATCH-Register erfolgen, Outputs ber PORT knnen seltsame Wirkungen zeigen
//Pin-Inputs mssen ber das PORT-Register erfolgen
#define KLine_Enable    LATBbits.LB4    //K-Line Transceiver Enable
#define KLine_TxPin     LATCbits.LC2    //K-Line Transmit - Bei nderung muss auch asm-Zugriff in KLine_Tx gendert werden
#define KLine_RxPin     PORTBbits.RB6   //K-Line Receive - Bei nderung muss auch asm-Zugriff in KLine_Rx gendert werden
#define Kons_Enable     LATCbits.LC5    //Konsole Enable
#define E_Tx_Pin        LATCbits.LC4    //Extra-Interface I/O Write-Register
#define E_Rx_Pin        PORTCbits.RC4   //Extra-Interface I/O Read-Register
#define Raw_Pin         PORTCbits.RC3   //Raw Select Schalter
#define Testpin         LATCbits.LC1    //Der Testpin, den man zum Debuggen toggeln kann
//Andere Bit-Definitionen
#define Kons_RxAvailable PIR1bits.RCIF      //Flag das UART-Empfang anzeigt
#define Kons_TxReady    TXSTAbits.TRMT      //Flag das anzeigt, das UART sendebereit ist


#include <xc.h>                 // Inkludiert automatisch richtigen Header fr im Projekt gewhlten Prozessor

//Prototypen fr KLine-Funktionen
void KLine_Init(void);
unsigned char KLine_RecvByte(void);
void KLine_RecvFrame(unsigned char anz);
void KLine_SendByte(unsigned char byte);
void KLine_SendFrame(unsigned char anz);
void baud10400(void);
void baud5200(void);
//Prototypen fr UART-Funktionen
void UART_Init(void);
unsigned char Kons_RecvByte(void);
void Kons_RecvFrame(unsigned char anz);
void Kons_SendByte(unsigned char byte);
void Kons_SendFrame(unsigned char anz);

//Prototypen fr andere Funktionen
void GPIO_Init(void);
void wait(unsigned char cnt);


//Globale Variablen, die auch im Assembler-Code verwendet werden
unsigned char m; 
unsigned char n; 
unsigned char dat; 
//Andere globale Variablen
unsigned char bformat;      //Format-Byte eines Frames
unsigned char brest[66];    //Die restlichen Bytes eines Frames, kann maximal 66 Bytes lang sein, max 63 Datenbytes plus 3 weitere

//Makros zur Bit-Manipulation/berprfung von Flag-Bits (Register-Bits und Pins gehen einfach ber entsprechende XXXbits.XXX)
#define SET_BIT(byte, bit) ((byte) |= (1UL << (bit)))           //Setzen eines Bits
#define CLEAR_BIT(byte,bit) ((byte) &= ~(1UL << (bit)))         //Lschen eines Bits
#define TOGGLE_BIT(byte,bit) ((byte) ^= (1UL << (bit)))         //Umschalten eines Bits
#define GET_BIT(byte,bit) (((byte) & (1UL << (bit))) >> (bit))  //Prfen eines Bits

    
void main(void) {

    unsigned char anz;
    unsigned char x;

    
    GPIO_Init();
    KLine_Init();
    UART_Init(); 
    Kons_RxAvailable=0;              //Evtl schon anstehenden Empfang vom Display lschen   
Testpin=0;
    while(1){               //Endlosschleife fr Anzeige-Update
        //Request vom Tester empfangen und an PC weiterleiten
        bformat = KLine_RecvByte();             //Wartet auf erstes Byte eines K-Line Frames
Testpin=1;
        if (bformat == 0 || bformat == 255) {   //0 kann nur nur bei einem Wake-up-Event vorkommen, 255 kommt bei einem Adapter-Check von Motoscan vor, da wird Kline fr einige us auf low gesetzt, was vom PIC als Startbit missverstanden wird
            Kons_SendByte(0x00);
        } else {
Testpin=0;
            anz = bformat & 0x3F;               //Ermittlung der Anzahl Datenbytes (Bits 5...0 im Formatbyte)
            anz = anz + 3;                      //Zustzlich zu den anz Datenbytes kommen noch 3 weitere Bytes (Target Address, Source Address und am Ende Checksum) 
for (x=0;x<anz;x++) {
    Testpin=1;
    Testpin=0;
}
            KLine_RecvFrame(anz);               //Restliche Bytes des Frames empfangen und nach brest schreiben
Testpin=1;
            Kons_SendByte(bformat);             //Erst Format-Byte an PC schicken...
            Kons_SendFrame(anz);                //...dann restlichen Frame
        }
Testpin=0;
        //Response vom PC empfangen und an Tester zurckleiten
        bformat = Kons_RecvByte();              //Wartet auf erstes Byte eines Response Frames
Testpin=1;
        if (bformat == 0) {                     //0x00 ist Response wenn 0x00 bermittelt wurde
            //Es geht nichts an den Tester zurck
        } else {
            anz = bformat & 0x3F;               //Ermittlung der Anzahl Datenbytes (Bits 5...0 im Formatbyte)
            anz = anz + 3;                      //Zustzlich zu den anz Datenbytes kommen noch 3 weitere Bytes (Target Address, Source Address und am Ende Checksum) 
            Kons_RecvFrame(anz);                //Restliche Bytes des Frames empfangen und nach brest schreiben
            KLine_SendByte(bformat);            //Erst Format-Byte an Tester schicken...
            KLine_SendFrame(anz);                //...dann restlichen Frame
        }   
Testpin=0;
    }
}

//Aufsetzen der GPIO Pins
void GPIO_Init() {
    Kons_Enable = 0;            //Display noch ausgeschaltet lassen
    TRISCbits.RC5 = 0;      //RC5(Display Enable)=Output
    TRISCbits.RC4 = 1;      //RC4(Extra-Interface)=Input
    TRISCbits.RC3 = 1;      //RC3(RawSelect)=Input    
    TRISCbits.RC1 = 0;      //RC1(Pin 15)=Output fr Debug-Zwecke toggeln 
}

//Aufsetzen der K-Line Pins
void KLine_Init() {
    KLine_Enable=0;                //K-Line Transceiver noch ausgeschaltet
    KLine_TxPin=1;                //K-Line Transceiver auf Recessive State
    TRISBbits.RB4 = 0;      //RB4(KLine Enable)=Output
    TRISCbits.RC2 = 0;      //RC2(KLine-Tx)=Output
    TRISBbits.RB6 = 1;      //RB6(KLine-Rx)=Input  
    //Transceiver einschalten, danach 2ms warten bis Transceiver ready ist (siehe Datenblatt Power-up to ready ist 1,5ms)
    KLine_Enable=1;
    wait(20);              
}

//Konfiguration des UART fr die Kommunikation mit der Konsole
void UART_Init() {
    TRISBbits.RB5 = 1;      //RB5(Rx)=Input
    TRISBbits.RB7 = 0;      //RB7(Tx)=Output
    ANSELHbits.ANS11 = 0;   //RB5(Rx) ist digital
    TXSTAbits.TX9 = 0;      //8-Bit
    TXSTAbits.TXEN = 1;     //Enable Transmit        
    TXSTAbits.SYNC = 0;     //Async Mode
    TXSTAbits.BRGH = 0;     //Baudrate Low Speed
    RCSTAbits.SPEN = 1;     //Enable Serial Port
    RCSTAbits.RX9 = 0;      //8-Bit
    RCSTAbits.CREN = 1;     //Enable reception bit Rx
    BAUDCONbits.DTRXP = 1;  //Invertierung zu active/idle low , wie vom PC-UART erzeugt wird
    BAUDCONbits.CKTXP = 1;  //Invertierung zu active/idle low, wie vom PC-UART erwartet wird  
    BAUDCONbits.BRG16 = 1;  //16-Bit Baudrate-Gernerierung
    SPBRGH = 0;             //16-Bit Baudzhler: BRG = (12MHz/(19200*16))-1 = 38
    SPBRG = 38; 
}

//Sendet ein Byte  ber den UART an die Konsole
void Kons_SendByte(unsigned char byte) {
    while(!Kons_TxReady); //Warte bis letzter Transmit beendet ist
    TXREG = byte;
}

//Sendet ein Frame mit anz Bytes aus brest an den PC
void Kons_SendFrame(unsigned char anz) {
    unsigned char x;
    for (x=0;x<anz;x++) {
        Kons_SendByte(brest[x]);
    }
}

//Wartet auf und empfngt ein Byte von der Konsole
unsigned char Kons_RecvByte() {
    while(!Kons_RxAvailable);         //Warte bis Byte empfangen wurde
    Kons_RxAvailable=0;
    return RCREG;
}

//Empfngt anz Bytes von der Konsole und schreibt diese nach brest
void Kons_RecvFrame(unsigned char anz) {
    unsigned char x;
    for (x=0;x<anz;x++) {
        brest[x] = Kons_RecvByte();
    }
}

//Warten auf fallende K-Line-Flanke und Empfang eines Datenbytes mit 10400 baud
unsigned char KLine_RecvByte(void) {
	__asm("_a0:  BTFSS	0x81,6,c");	//Zuerst muss KL-Rx=high sein um zu verhindern, dass bereits anstehendes Low (z.B. 25ms Low im Wake-up Event) gleich einen Empfang auslst
	__asm("GOTO     _a0");          //Nein: Weiterwarten    
	__asm("_a1:  BTFSC	0x81,6,c");	//Startbit (Low=0V) - fallende Flanke auf KL-Rx? PORTB,6(0x81,6) ist KL_Rx
	__asm("GOTO     _a1");          //Nein: Weiterwarten
	__asm("CLRF     _dat");
	__asm("MOVLW	8");
	__asm("MOVWF	_n");
	__asm("NOP");
	__asm("NOP");
	__asm("NOP");
	__asm("NOP");
	baud5200();                 //Startbit wurde registriert, warte 1/2 baudzeit (danach gleich noch eine ganze) = 1,5 baud seit der Startbitflanke = Mitte vom ersten bit
	__asm("_a2: NOP");
	baud10400();                //Warte volle baudrate-10 Zyklen (werden in Schleife verbraten), um in naechstes Bit zu kommen
	__asm("BTFSS	0x81,6,c"); //Pruefe Bit PORTB,6(0x81,6) ist KL_Rx
	__asm("GOTO     _a3");		//=0V: (=Logisch 0)
	__asm("BSF		status,0,c");	//=5V: (=Logisch 1) Carry mit 1 besetzen
	__asm("GOTO     _a4");
	__asm("_a3: BCF		status,0,c");	//Carry mit 0 besetzen
	__asm("NOP");				//Damit gleich lang wie BSF-Zweig
    __asm("NOP");
	__asm("_a4: RRCF		_dat,1");	//Carry in Byte verschieben (K-Line:LSB first -> Bits werden von links nach rechts reingeschoben)
	__asm("DECFSZ	_n,1");
	__asm("GOTO     _a2");
    return dat;
}

//Empfngt anz Bytes aus dem KLine-Interface und schreibt diese in den brest-Buffer
void KLine_RecvFrame(unsigned char anz) {
    //Aufzurufen mit z.B. fr 10 erwartete Bytes: "unsigned char frame[10]; KLine_RecvFrame(frame, sizeof(frame));"
    unsigned char x;
    for (x=0;x<anz;x++) {
        brest[x] = KLine_RecvByte();
    }
}

//Sendet ein Byte in das KLine-Interface
void KLine_SendByte(unsigned char byte) {
    dat=byte;
	__asm("MOVLW 	8");
	__asm("MOVWF 	_n");				//8 Bits senden
	__asm("BCF   	0x8b,2,c");			//Startbit setzen (Low = 0V) LATC,2(0x8b,2) ist KL_Tx, zum Schreiben NICHT PORTC,2(0x82,2) verwenden!
	__asm("NOP");
	__asm("NOP");
	__asm("NOP");
	__asm("NOP");
	__asm("NOP");
	baud10400();			//Baudzeit warten
	__asm("_a9: RRCF   	_dat,1");			//Bits durchschieben (K-Line:LSB first -> Bits werden von links nach rechts rausgeschoben)
	__asm("BTFSS 	status,0,c");			//Bit checken = 1?
	__asm("GOTO     _aa");				//nein: Sende 0
	__asm("NOP");						//ja: Sende 1
	__asm("BSF	  	0x8b,2,c");			//High=5V
	__asm("GOTO     _ab");				//Ab zur Pause 
    __asm("_aa: NOP");
	__asm("BCF   	0x8b,2,c");			//Low=0V
	__asm("NOP");
	__asm("NOP");
    __asm("_ab:");
	baud10400();			//Baudzeit warten
	__asm("DECFSZ 	_n,1");				//n=n-1, n=0 (Byte zu Ende)?
	__asm("GOTO  	_a9");			//nein: weiter mit naechstem Bit
	__asm("NOP");						//ja: Stopbit setzen
	__asm("NOP");
	__asm("NOP");
	__asm("NOP");
	__asm("NOP");
	__asm("BSF   	0x8b,2,c");			//Stopbit setzen (High = 5V)
	baud10400();
}

//Sendet einen Frame von anz Bytes aus brest in das KLine-Interface
void KLine_SendFrame(unsigned char anz) {
    unsigned char x;
    for (x=0;x<anz;x++) {
        KLine_SendByte(brest[x]);
    }
}

//Warteschleife fuer Baudrate 10400
void baud10400(void) {
    //Baudrate 10400 (=96us) bei 12MHz/4 Takt (=288 Zyklen)
    //11 Zyklen werden schon in der Rx-Routine zwischen jedes Bit gesetzt d.h. zusammen mit aufrufendem CALL werden hier 277 Zyklen verbraten
    //2+2+(90*3)+1+2 (CALL + 2*MOV + 90*Schleife + NOP + RETURN)
    __asm("MOVLW  	90");	
	__asm("MOVWF  	_m");
	__asm("_a5: DECFSZ 	_m,1");			
	__asm("GOTO  	_a5");
    __asm("NOP");
}

//Warteschleife fuer 1/2 Baudrate 10400
void baud5200(void) {
    //Eine halbe Baudzeit bei 10400 baud (=48us) bei 12MHz/4 Takt (=144 Zyklen)
    ////2+2+(46*3)+2 (CALL + 2*MOV + 46*Schleife + RETURN)
    __asm("MOVLW  	46");	
	__asm("MOVWF  	_m");
	__asm("_a6: DECFSZ 	_m,1");			
	__asm("GOTO  	_a6");   
}

//Warteschleife fuer cnt * 0,1ms
void wait(unsigned char cnt) {
    //Innere m-Schleife M dauert 3+98*3=297 Zyklen (knapp 0,1ms). Auere n-Schleife wiederholt M je nach cnt
    n=cnt;
    __asm("MOVWF    _n");
    __asm("_a7: MOVLW  	98");	
	__asm("MOVWF  	_m");
    __asm("NOP");
	__asm("_a8: DECFSZ 	_m,1");			
	__asm("GOTO  	_a8");
    __asm("DECFSZ 	_n,1");			
	__asm("GOTO  	_a7");
}

