// F800 Motcom-Zentrale
const unsigned char ZVER = 0x03;    //Versionsnummer der SW. Obere 4-Bit sind 10er Dezimal, untere 4 Bit sind 1er-Deszimal. D.h. Version 26 wre also 0x26 
/* Fragt ber K-Line verschiedene Parameter aus dem Fahrzeugbus ab, generiert daraus Bildschirmanzeigen und bertrgt diese zur Konsole
 * RB4, Pin13, Output, K-Line Transceiver Enable
 * RB6, Pin11, Input,  K-Line Receive
 * RC2, Pin14, Output, K-Line Transmit
 * UTX, Pin10, Output, Transmit zur Konsole
 * URX, Pin12, Input,  Receive von Konsole
 * RC5, Pin5,  Output, Display Enable
 * RC4, Pin6,  I/O,    Extra Interface (Bidirectional)
 * RC3, Pin7,  Input,  Raw Select Schalter
 * RC1, Pin15, Output, Debug-Pin, ist ansonsten frei, kann benutzt werden um durch Toggeln spez. Aktivitt anzuzeigen
 * Takt = 12MHz : Externer Quarz
 * K-Line Receive luft auf 10400 baud
 * UART zur Konsole luft mit 9600 baud, von den Pegeln her ist das Interface so ausgelegt, dass es zu Testzwecken auch mit einem normalen PC-RS232 angesteuert werden kann
 */

//Configuration-Bits - erstellt ber "Set Configuration Bits" - "Generate Source Code"
// CONFIG1L
#pragma config CPUDIV = NOCLKDIV// CPU System Clock Selection bits (No CPU System Clock divide)
#pragma config USBDIV = OFF     // USB Clock Selection bit (USB clock comes directly from the OSC1/OSC2 oscillator block; no divide)
// CONFIG1H
#pragma config FOSC = HS        // Oscillator Selection bits (HS oscillator) - XT ging beim F13K50 aber nicht mehr beim F14K50
#pragma config PLLEN = OFF      // 4 X PLL Enable bit (PLL is under software control)
#pragma config PCLKEN = ON      // Primary Clock Enable bit (Primary clock enabled)
#pragma config FCMEN = OFF      // Fail-Safe Clock Monitor Enable (Fail-Safe Clock Monitor disabled)
#pragma config IESO = OFF       // Internal/External Oscillator Switchover bit (Oscillator Switchover mode disabled)
// CONFIG2L
#pragma config PWRTEN = ON      // Power-up Timer Enable bit (PWRT enabled)
#pragma config BOREN = OFF      // Brown-out Reset Enable bits (Brown-out Reset disabled in hardware and software)
#pragma config BORV = 19        // Brown-out Reset Voltage bits (VBOR set to 1.9 V nominal)
// CONFIG2H
#pragma config WDTEN = OFF      // Watchdog Timer Enable bit (WDT is controlled by SWDTEN bit of the WDTCON register)
#pragma config WDTPS = 32768    // Watchdog Timer Postscale Select bits (1:32768)
// CONFIG3H
#pragma config HFOFST = OFF     // HFINTOSC Fast Start-up bit (The system clock is held off until the HFINTOSC is stable.)
#pragma config MCLRE = OFF      // MCLR Pin Enable bit (RA3 input pin enabled; MCLR disabled)
// CONFIG4L
#pragma config STVREN = ON      // Stack Full/Underflow Reset Enable bit (Stack full/underflow will cause Reset)
#pragma config LVP = ON         // Single-Supply ICSP Enable bit (Single-Supply ICSP enabled)
#pragma config BBSIZ = OFF      // Boot Block Size Select bit (512W boot block size)
#pragma config XINST = OFF      // Extended Instruction Set Enable bit (Instruction set extension and Indexed Addressing mode disabled (Legacy mode))
// CONFIG5L
#pragma config CP0 = OFF        // Code Protection bit (Block 0 not code-protected)
#pragma config CP1 = OFF        // Code Protection bit (Block 1 not code-protected)
// CONFIG5H
#pragma config CPB = OFF        // Boot Block Code Protection bit (Boot block not code-protected)
#pragma config CPD = OFF        // Data EEPROM Code Protection bit (Data EEPROM not code-protected)
// CONFIG6L
#pragma config WRT0 = OFF       // Table Write Protection bit (Block 0 not write-protected)
#pragma config WRT1 = OFF       // Table Write Protection bit (Block 1 not write-protected)
// CONFIG6H
#pragma config WRTC = OFF       // Configuration Register Write Protection bit (Configuration registers not write-protected)
#pragma config WRTB = OFF       // Boot Block Write Protection bit (Boot block not write-protected)
#pragma config WRTD = OFF       // Data EEPROM Write Protection bit (Data EEPROM not write-protected)
// CONFIG7L
#pragma config EBTR0 = OFF      // Table Read Protection bit (Block 0 not protected from table reads executed in other blocks)
#pragma config EBTR1 = OFF      // Table Read Protection bit (Block 1 not protected from table reads executed in other blocks)
// CONFIG7H
#pragma config EBTRB = OFF      // Boot Block Table Read Protection bit (Boot block not protected from table reads executed in other blocks)

//Pin-Definitionen
//Pin-Outputs sollten ber das LATCH-Register erfolgen, Outputs ber PORT knnen seltsame Wirkungen zeigen
//Pin-Inputs mssen ber das PORT-Register erfolgen
#define KLine_Enable    LATBbits.LB4    //K-Line Transceiver Enable
#define KLine_TxPin     LATCbits.LC2    //K-Line Transmit - Bei nderung muss auch asm-Zugriff in KLine_Tx gendert werden
#define KLine_RxPin     PORTBbits.RB6   //K-Line Receive - Bei nderung muss auch asm-Zugriff in KLine_Rx gendert werden
#define Kons_Enable     LATCbits.LC5    //Konsole Enable
#define E_Tx_Pin        LATCbits.LC4    //Extra-Interface I/O Write-Register
#define E_Rx_Pin        PORTCbits.RC4   //Extra-Interface I/O Read-Register
#define Raw_Pin         PORTCbits.RC3   //Raw Select Schalter
#define Shutdown        PORTCbits.RC0   //Shutdown-Signal, INT0-Pin
#define Testpin         LATCbits.LC1    //Der Testpin, den man zum Debuggen toggeln kann
//Andere Bit-Definitionen
#define Kons_RxAvailable PIR1bits.RCIF      //Flag das UART-Empfang anzeigt
#define Kons_TxReady    TXSTAbits.TRMT      //Flag das anzeigt, das UART sendebereit ist
#define T0_Overflow     INTCONbits.TMR0IF   //Flag das anzeigt wenn Timer0 bergelaufen ist
#define T0_Enable       T0CONbits.TMR0ON    //Enable des Timer0

//K-Line Request-Definitionen
#define KL_ABSSTARTCOM  {0x81, 0x29, 0xF1, 0x81, 0x1C}
#define KL_DWASTARTCOM  {0x81, 0x41, 0xF1, 0x81, 0x34}
#define KL_ECUSTARTCOM  {0x81, 0x12, 0xF1, 0x81, 0x05}
#define KL_KOMSTARTCOM  {0x81, 0x60, 0xF1, 0x81, 0x53}
#define KL_ZFESTARTCOM  {0x81, 0x72, 0xF1, 0x81, 0x65}
#define KL_ABSGETDTC    {0x84, 0x29, 0xF1, 0x18, 0x02, 0xFF, 0xFF, 0xB6}
#define KL_DWAGETDTC    {0x84, 0x41, 0xF1, 0x18, 0x02, 0xFF, 0xFF, 0xCE}
#define KL_ECUGETDTC    {0x84, 0x12, 0xF1, 0x18, 0x02, 0xFF, 0xFF, 0x9F}
#define KL_KOMGETDTC    {0x84, 0x60, 0xF1, 0x18, 0x02, 0xFF, 0xFF, 0xED}
#define KL_ZFEGETDTC    {0x84, 0x72, 0xF1, 0x18, 0x02, 0xFF, 0xFF, 0xFF}
#define KL_ECUDATA1     {0x83, 0x12, 0xF1, 0x22, 0x40, 0x00, 0xE8}
#define KL_ECUDATA2     {0x83, 0x12, 0xF1, 0x22, 0x40, 0x11, 0xF9}
#define KL_ECUDATAD     {0x83, 0x12, 0xF1, 0x22, 0x40, 0x05, 0xED}
#define KL_KOMDATA      {0x82, 0x60, 0xF1, 0x21, 0x04, 0xF8}

//Definitionen der Screen-Nummern
#define S_Licht     0
#define S_Geschw    1
#define S_DrehzNum  2
#define S_DrehzBar  3
#define S_Distanz   4
#define S_Zeit      5
#define S_VAvg      6
#define S_Reset     7
#define S_Mottemp   8
#define S_Lufttemp  9
#define S_Luftdrk   10
#define S_VBat      11
#define S_Benzin    12
#define S_BzDruck   13
#define S_ZWinkel   14
#define S_Dklappe   15
#define S_Drehmnt   16
#define S_Leistung  17
#define S_LMasse    18
#define S_Lambda    19
#define S_DTC_JN    20
#define S_DTC_Ovr   21
#define S_DTC_Det   22
#define S_DoConnect 23
#define S_DoReset   24
#define S_DoLicht   25
#define S_TgConnect 26


#include <xc.h>                 // Inkludiert automatisch richtigen Header fr im Projekt gewhlten Prozessor
#include "zeichensatz.h"        //Der Zeichensatz des Displays in der Konsole

//EEPROM-Initialisierung mit Grundwerten
 //3 Bits 0..2 = Initiale Kilometer wie sie 2025 waren (00-E3-21 = 58.145)
 //4 Bits 3..6 = Initiale Sekunden wie sie 2025 waren (00-A6-39-1B = 10.893.595)
 //1 Bit 7 = leer (__EEPROM_DATA - Makro bentigt genau 8 Parameter)
 __EEPROM_DATA(0x00, 0xE3, 0x21, 0x00, 0xA6, 0x39, 0x1B, 0x00);

//Prototypen fr KLine-Funktionen
void KLine_Init(void);
unsigned char KLine_RecvByte(void);
void KLine_RecvFrame(unsigned char *bytes, unsigned char count);
void KLine_SendByte(unsigned char byte);
void KLine_SendFrame(unsigned char bytes[], unsigned char len);
void KLine_SendWakeup(void);
void baud10400(void);
void baud5200(void);
//Prototypen fr UART-Funktionen
void UART_Init(void);
unsigned char Kons_SendWord(unsigned char cont, unsigned char data);
unsigned char Kons_SendFrame(unsigned char DChar[16]);
unsigned char Kons_RecvResp(void);
//Prototypen fr andere Funktionen
void GPIO_Init(void);
void TMR0_Init(void);
void TMR0_Reset(void);
void wait(unsigned char cnt);
void wait55ms(void);
void wait200ms(void);
void INT_Init(void);
unsigned char EE_ReadByte(unsigned char adr);
void EE_WriteByte(unsigned char adr, unsigned char dat);
void AddToCols(unsigned char value);
//void __interrupt(high_priority) ISR(void);
//Prototypen fr Menuefunktionen
unsigned char setScreenID(unsigned char tasten, unsigned char ohne, unsigned char oben, unsigned char unten, unsigned char beide);
unsigned char GetChar(unsigned short value, unsigned short ziffer);
unsigned char getGang(unsigned char gang);
unsigned char makeLichtauswahl(void);
unsigned char makeGeschwindigkeit(void);
unsigned char makeDrehzahlNum(void);
unsigned char makeDrehzahlBar(void);
unsigned char makeDistanz(void);
unsigned char makeFahrzeit(void);
unsigned char makeVDurchschnitt(void);
unsigned char makeReset(void);
unsigned char makeKuehltemp(void);
unsigned char makeAussentemp(void);
unsigned char makeLuftdruck(void);
unsigned char makeSpannung(void);
unsigned char makeKraftstoff(void);
unsigned char makeSpritzdruck(void);
unsigned char makeZuendwinkel(void);
unsigned char makeDrosselklappe(void);
unsigned char makeDrehmoment(void);
unsigned char makeLeistung(void);
unsigned char makeLuftmasse(void);
unsigned char makeLambda(void);
unsigned char makeConnect(void);
unsigned char makeDTC1(void);
unsigned char makeDTC2(void);
unsigned char makeDTC3(void);

//Globale Variablen, die auch im Assembler-Code verwendet werden
unsigned char m; 
unsigned char n; 
unsigned char dat; 
unsigned char i_kilometerH; //Initiale Kilometer Highbyte
unsigned char i_kilometerM; //Initiale Kilometer Midbyte
unsigned char i_kilometerL; //Initiale Kilometer Lowbyte
unsigned char i_sekundenH;  //Initiale Sekunden Highbyte
unsigned char i_sekundenMH; //Initiale Sekunden MidHighbyte
unsigned char i_sekundenML; //Initiale Sekunden MidLowbyte
unsigned char i_sekundenL;  //Initiale Sekunden Lowbyte
//Andere globale Variablen
unsigned char screenID;     //Die ID des aktuell anzuzeigenden Screens
unsigned char screenIDAlt;  //Die ID des zuletzt angezeigten Screens (wichtig fr die Rekonstruktion der ScreenID nach Dummy-Screens)
unsigned short ergebnis;    //16-Bit Ergebniswert von Berechnungen - zur reihenweisen Extrahierung der Dezimalziffern
unsigned char klinercv;     //Zhler fr Einholung der Zusatzdaten
unsigned char klineresp[67]; //Der Buffer fr alle Responses ber K-Line
unsigned char d_kraftstoff; //Restkraftstoff
unsigned char d_geschw;     //Geschwindigkeit
unsigned char d_drehzahlH;  //Drehzahl Highbyte
unsigned char d_drehzahlL;  //Drehzahl Lowbyte
unsigned char d_lufttemp;   //Lufttemperatur
unsigned char d_motortemp;  //Motortemperatur
unsigned char d_zundwinkel; //Zndwinkel
unsigned char d_dkwinkel;   //Drosselklappenwinkel
unsigned char d_motmoment;  //Motormoment
unsigned char d_spannung;   //Batteriespannung
unsigned char d_kilometerH; //Kilometer Highbyte
unsigned char d_kilometerM; //Kilometer Midbyte
unsigned char d_kilometerL; //Kilometer Lowbyte
unsigned char d_sekundenH;  //Sekunden Highbyte
unsigned char d_sekundenMH; //Sekunden MidHighbyte
unsigned char d_sekundenML; //Sekunden MidLowbyte
unsigned char d_sekundenL;  //Sekunden Lowbyte
unsigned char d_luftdruckH; //Luftdruck Highbyte
unsigned char d_luftdruckL; //Luftdruck Lowbyte
unsigned char d_gang;       //Gang
unsigned char d_benzdruckH; //Kraftstoffdruck Highbyte
unsigned char d_benzdruckL; //Kraftstoffdruck Lowbyte
unsigned char d_luftmasseH; //Luftmasse Highbyte
unsigned char d_luftmasseL; //Luftmasse Lowbyte
unsigned char d_lambda;     //Lambda
unsigned char d_dig;        //Aktivitt Motorlfter und Kraftstoffpumpe
#define d_luefter 1         //Bit in d_dig, das anzeigt ob der Motorlfter luft
#define d_bpumpe 2          //Bit in d_dig, das anzeigt ob die Benzinpumpe luft
unsigned char dtc_abs;      //Anzahl DTCs im ABS-Gert
unsigned char dtc_dwa;      //Anzahl DTCs im DWA-Gert
unsigned char dtc_kom;      //Anzahl DTCs im KOMBI-Gert
unsigned char dtc_ecu;      //Anzahl DTCs im MOTECU-Gert
unsigned char dtc_zfe;      //Anzahl DTCs im ZFE-Gert
unsigned char DTCnum;       //Aktuell anzuzeigender DTC-Screen
unsigned char tcola[4];     //Collector-Buffer A fr Khlwassertemperaturen 
unsigned char pcola;        //Akt-Pointer im Collector-Buffer A
unsigned char tcolb[4];     //Collector-Buffer B fr Khlwassertemperaturen 
unsigned char pcolb;        //Akt-Pointer im Collector-Buffer B

unsigned char flags;           
#define f_connected 0              //Flag das anzeigt, ob MotCom aktuell mit dem Motorrad verbunden ist
#define f_lichtwunsch 1            //Der allgemeine Beleuchtungszustands-Wunsch (kann unterschiedlich sein zu Status. Z.B. Wenn gerade der Lichtauswahl-Screen angezeigt wird, ist der Status immer gleich Ein, auch wenn der allgemeine Lichtwunsch z.b. fr die anderen Screens gleich Aus ist)
#define f_lichtstatus 2            //Der Zustand in dem die Beleuchtung aktuell gerade sein soll
#define f_lze 3                    //Indikator dass Leading-Zero-Elimination noch aktiv ist
#define f_prop 4                   //Der aktuelle Anzeige Zustand von bewegten Objekten (z.B. drehender Propeller) - so dass dieser mit jedem Anzeige-Update neu gezeichnet wird 
#define f_colbfull 5               //Indikator, dass Collector-Buffer B fr Khlwassertemperaturen vollstndig gefllt ist
#define f_dtcda 6                  //Indikator, dass DTCs abgefragt wurden - fr Anzeigr makeDTC2
unsigned char timeout;
#define to_kons 0                   //Timeout bei Receive von Konsole
#define to_kline 1                  //Timeout bei Receive von K-Line
#define to_kl_ecud 2                //Timeout bei Einholen der Daten von ECU-Digital
#define to_kl_ecua1 3               //Timeout bei Einholen der Daten von ECU-Analog1
#define to_kl_ecua2 4               //Timeout bei Einholen der Daten von ECU-Analog2
#define to_kl_kombi 5               //Timeout bei Einholen der Daten von KOMBI

#define Ch_nc Ch_Minus              //Der Standard-Character fr Anzeigewerte wenn nicht Connected ist

//Makros zur Bit-Manipulation/berprfung von Flag-Bits (Register-Bits und Pins gehen einfach ber entsprechende XXXbits.XXX)
#define SET_BIT(byte, bit) ((byte) |= (1UL << (bit)))           //Setzen eines Bits
#define CLEAR_BIT(byte,bit) ((byte) &= ~(1UL << (bit)))         //Lschen eines Bits
#define TOGGLE_BIT(byte,bit) ((byte) ^= (1UL << (bit)))         //Umschalten eines Bits
#define GET_BIT(byte,bit) (((byte) & (1UL << (bit))) >> (bit))  //Prfen eines Bits


//Nach debug und vor Fertigstellung zu prfen:
//- Ist Testsequenz kommentiert?
//- 2x while(!Kons_RxAvailable); im Startup auskommentiert?
//- Kons_RecvResp nicht mehr im Debug-Modus? (Debug-Modus-Zeile kommentiert)
//- KLine_RecvFrame nicht mehr im Debug-Modus? (Debug-Modus-Zeilen kommentiert, non Debug-Modus-Zeile auskommentiert)
    
void main(void) {
    unsigned char ret;    
    GPIO_Init();
    KLine_Init();
    UART_Init(); 
    TMR0_Init();  
//    #include "testsequenz.h"        //Testsequenz fr HW-Routinen   
    //Laden des EEPROM-Stands (z.B. Distanz, Fahrzeit) 
    i_kilometerH = EE_ReadByte(0);
    i_kilometerM = EE_ReadByte(1); 
    i_kilometerL = EE_ReadByte(2); 
    i_sekundenH = EE_ReadByte(3);
    i_sekundenMH = EE_ReadByte(4); 
    i_sekundenML = EE_ReadByte(5); 
    i_sekundenL = EE_ReadByte(6);   
    //Konsole einschalten und initialisieren
    Kons_Enable=1;              //Konsole einschalten - Display initialisiert sich nun - wenn das beendet ist, meldet die Konsole Empfangsbereitschaft durch Senden eines Bytes (0x00)
    ret=RCREG;                  //Evtl schon anstehenden Empfang vom Display lschen (setzt Kons_RxAvailable auf 0)
    while(!Kons_RxAvailable);   //Wartet auf Besttigungs-Meldung von Konsole (sollte 0x00 sein, wird aber nicht geprft), dass Konsole gestartet ist und nun Versionsnummer von Zentrale braucht - nicht ber Kons_RecvResp() machen, da hier kein Timeout greifen soll
    while(!Kons_TxReady);        //Warte bis letzter Transmit beendet ist (sollte es eigentlich nicht geben)
    TXREG = ZVER;           //Versionsnummer an Konsole senden
                            //Konsole initialisiert sich nun weiter, macht die Generierung der Sonderzeichen, zeigt Splash-Screen an bis Taste gedrckt ist. Nach Tastendruck meldet die Konsole Empfangsbereitschaft durch Senden eines Bytes (0xFF)
    ret=RCREG;                  //Evtl schon anstehenden Empfang vom Display lschen (setzt Kons_RxAvailable auf 0)
    while(!Kons_RxAvailable);   //Wartet auf Ready-Meldung vom Display nach Wegdrcken des Splash-Screens (sollte 0xFF sein, wird aber nicht geprft) - nicht ber Kons_RecvResp() machen, da hier kein Timeout greifen soll
    wait(1);                //0,1ms warten um Konsole Zeit zu geben auf Empfangsbereitschaft zu gehen
    INT_Init();             //Interrupts konfigurieren und einschalten (z.B. Shutdown-Interrupt)
    //In normalen Anzeige-Betrieb gehen
//    screenID = S_DoConnect;         //Auto-Connect: Erste Anzeige soll der Connect-Screen sein
    screenID = S_Licht;           //Manueller Connect: Erste Anzeige soll der Licht-Screen sein
    T0_Enable = 1;          //Display-Updaterate-Timer einschalten
    while(1){               //Endlosschleife fr Anzeige-Update
        switch(screenID) {
            case S_Licht:         //Lichtauswahl
                SET_BIT(flags,f_lichtstatus);              //lichtstatus = 1: Licht soll in diesem Screen immer an sein             
                ret = makeLichtauswahl();                       
                screenID = setScreenID(ret,S_Licht,S_Geschw,S_DoLicht,S_TgConnect);    
                if (!GET_BIT(flags,f_lichtwunsch)) CLEAR_BIT(flags,f_lichtstatus);  //lichtstatus = lichtwunsch: Licht zurck auf gewnschten Status
                break;
            case S_Geschw:         //Geschwindigkeit
                ret = makeGeschwindigkeit();            //Holt entsprechende Daten aus dem Motorrad, rstellt den Screen, sendet ihn an die Konsole und empfngt den Tastenstatus zurck
                screenID = setScreenID(ret,S_Geschw,S_Distanz,S_DrehzNum,S_TgConnect);  //Ermittelt, welcher Screen als nchstes angezeigt werden soll, abhngig von dem Tastenstatus (keine, oben, unten, beide)
                break;
            case S_DrehzNum:         //Drehzahl numerisch
                ret = makeDrehzahlNum();
                screenID = setScreenID(ret,S_DrehzNum,S_Distanz,S_DrehzBar,S_TgConnect);
                break;
            case S_DrehzBar:         //Drehzahl Balkenanzeige
                ret = makeDrehzahlBar();
                screenID = setScreenID(ret,S_DrehzBar,S_Distanz,S_Geschw,S_TgConnect);
                break;
            case S_Distanz:         //Distanz
                ret = makeDistanz();
                screenID = setScreenID(ret,S_Distanz,S_Mottemp,S_Zeit,S_TgConnect);
                break;
            case S_Zeit:         //Fahrzeit
                ret = makeFahrzeit();
                screenID = setScreenID(ret,S_Zeit,S_Mottemp,S_VAvg,S_TgConnect);
                break;
            case S_VAvg:         //Durchschnittsgeschwindigkeit
                ret = makeVDurchschnitt();
                screenID = setScreenID(ret,S_VAvg,S_Mottemp,S_Reset,S_TgConnect);
                break;
            case S_Reset:         //Reset-Auswahl
                ret = makeReset();
                screenID = setScreenID(ret,S_Reset,S_DoReset,S_Distanz,S_TgConnect);
                break;
            case S_Mottemp:         //Khlmittel-Temperatur
                ret = makeKuehltemp();
                screenID = setScreenID(ret,S_Mottemp,S_Benzin,S_Lufttemp,S_TgConnect);
                break;
            case S_Lufttemp:         //Auentemperatur-Temperatur
                ret = makeAussentemp();
                screenID = setScreenID(ret,S_Lufttemp,S_Benzin,S_Luftdrk,S_TgConnect);
                break;
            case S_Luftdrk:        //Luftdruck
                ret = makeLuftdruck();
                screenID = setScreenID(ret,S_Luftdrk,S_Benzin,S_VBat,S_TgConnect);
                break;
            case S_VBat:        //Bordspannung
                ret = makeSpannung();
                screenID = setScreenID(ret,S_VBat,S_Benzin,S_Mottemp,S_TgConnect);
                break;
            case S_Benzin:        //Restkraftstoffmenge
                ret = makeKraftstoff();
                screenID = setScreenID(ret,S_Benzin,S_DTC_JN,S_BzDruck,S_TgConnect);
                break;
            case S_BzDruck:        //Einspritzdruck
                ret = makeSpritzdruck();
                screenID = setScreenID(ret,S_BzDruck,S_DTC_JN,S_ZWinkel,S_TgConnect);
                break;
            case S_ZWinkel:        //Zndwinkel
                ret = makeZuendwinkel();
                screenID = setScreenID(ret,S_ZWinkel,S_DTC_JN,S_Dklappe,S_TgConnect);
                break;    
            case S_Dklappe:        //Drosselklappenwinkel
                ret = makeDrosselklappe();
                screenID = setScreenID(ret,S_Dklappe,S_DTC_JN,S_Drehmnt,S_TgConnect);
                break;
            case S_Drehmnt:        //Drehmoment
                ret = makeDrehmoment();
                screenID = setScreenID(ret,S_Drehmnt,S_DTC_JN,S_Leistung,S_TgConnect);
                break;
            case S_Leistung:        //Leistung
                ret = makeLeistung();
                screenID = setScreenID(ret,S_Leistung,S_DTC_JN,S_LMasse,S_TgConnect);
                break;
            case S_LMasse:        //Luftmasse
                ret = makeLuftmasse();
                screenID = setScreenID(ret,S_LMasse,S_DTC_JN,S_Lambda,S_TgConnect);
                break;               
            case S_Lambda:        //Lambda
                ret = makeLambda();
                screenID = setScreenID(ret,S_Lambda,S_DTC_JN,S_Benzin,S_TgConnect);
                break;            
            case S_DTC_JN:        //DTCs lesen Ja oder Nein?
                ret = makeDTC1();
                screenID = setScreenID(ret,S_DTC_JN,S_Licht,S_DTC_Ovr,S_DTC_JN);    //Hier mit Doppeltaste nicht in den Connect-Screen wechseln
                break;
            case S_DTC_Ovr:        //DTCs einholen und bersicht anzeigen
                ret = makeDTC2();
                screenID = setScreenID(ret,S_DTC_Ovr,S_Licht,S_DTC_Det,S_DTC_Ovr); //Hier mit Doppeltaste nicht in den Connect-Screen wechseln
                break;
            case S_DTC_Det:        //Fr jedes Gert nacheinander die entsprechende Anzahl der DTCs anzeigen
                ret = makeDTC3();
                if ((ret & 0x03) == 2) DTCnum--;        //Wenn untere Taste gedrckt wurde: im nchsten makeDTC3 den nchsten DTC anzeigen
                screenID = setScreenID(ret,S_DTC_Det,S_Licht,S_DTC_Det,S_DTC_Det); //Hier mit Doppeltaste nicht in den Connect-Screen wechseln
                break;
            case S_DoConnect:       //Wakeup und Verbindung mit den ECUs des Motorrads             
                ret = makeConnect();                    //Verbindung mit Motorrad herstellen - return erst nach erfolgter Verbindung (keine Tastendrucke)                 
                screenID = screenIDAlt;                 //Nach Verbindung zurck zu letztem Screen
                break;
            case S_DoReset:       //Dummy-Screen ohne eigene Anzeige. Wird fr Reset der Fahrdaten verwendet.
                i_kilometerH = d_kilometerH;  //Alle Initialwerte auf die aktuellen Werte setzen
                i_kilometerM = d_kilometerM; 
                i_kilometerL = d_kilometerL; 
                i_sekundenH = d_sekundenH;
                i_sekundenMH = d_sekundenMH; 
                i_sekundenML = d_sekundenML; 
                i_sekundenL = d_sekundenL; 
                screenID = S_Distanz;       //Nach Reset zurck zum Distanz-Screen
                break;
            case S_DoLicht:       //Dummy-Screen ohne eigene Anzeige. Wird fr das Umschalten des Lichts verwendet.
                TOGGLE_BIT(flags,f_lichtwunsch);
                screenID = screenIDAlt;
                break;
            case S_TgConnect:       //Dummy-Screen ohne eigene Anzeige. Wird fr das Connecten/Disconnected verwendet.
                if (GET_BIT(flags, f_connected)) {      //Wenn aktuell connected ist: disconnect
                    CLEAR_BIT(flags, f_connected);
                    SET_BIT(timeout, to_kl_ecua1);      //Alle Timeout-Flags setzen - damit in den Anzeigen das Ch_nc angezeigt wird
                    SET_BIT(timeout, to_kl_ecua2);
                    SET_BIT(timeout, to_kl_ecud);
                    SET_BIT(timeout, to_kl_kombi);
                    screenID = screenIDAlt;
                } else {                                //Wenn aktuell disconnected ist: connect
                    screenID = S_DoConnect;             //Im connect-screen wird bei Erfolg auch connected flag gesetzt, die einzelnen timeout-flags werden dann bei der ersten Dateneinholung gesetzt
                }
                break;
            default:        //Nicht existierende screenID
                screenID = S_Licht;   //Zurck zu Lichtauswahl
        }    
        //Bis hierher sind fr Konsoleupdate (36ms) und Konsoleresponse (1ms) 37ms vergangen - ab hier Einholen der Daten ber K-Line
        if (GET_BIT(flags, f_connected)) {      //Daten aus den ECUs einholen, wenn connected ist
            //Fr Einholen der Daten vergehen weitere max. 424ms-440ms, d.h. Schleifenlnge muss insgesamt also mindestens 477ms sein
            //- 188ms: Einholen ECU-AusgangAnalog1: (siehe Messungen, typisch 94ms, max 188ms : 7ms fr 7 Byte-Request, ca. 31ms ECU-Delay, 67ms fr 67 Byte-Response)
            //- 55ms: Zwangsdelay P3 (siehe ISO 14230-2)
            //- 79ms: Einholen ECU-AusgangAnalog2: (siehe Messungen), typisch 60ms, max 79ms : 7ms fr 7 Byte-Request, ca. 31ms ECU-Delay, 22ms fr 22 Byte-Response)
            //- 55ms: Zwangsdelay P3 (siehe ISO 14230-2)   
            //Plus noch eine der folgenden (entweder/oder))
            //- 63ms: Einholen ECU-AusgangDigital: (siehe Messungen, typisch 47ms, max 63ms : 7ms fr 7 Byte-Request, ca. 31ms ECU-Delay, 12ms fr 12 Byte-Response)
            //- 47ms: Einholen KOMBI-Kraftstoff: (siehe Messungen), typisch 53ms, max 47ms : 6ms fr 6 Byte-Request, ca. 31ms ECU-Delay, 16ms fr 16 Byte-Response)
            unsigned char e1[] = KL_ECUDATA1;     //Senden des Request fr ECU-AusgangAnalog1 - Response sind 67 Byte
            KLine_SendFrame(e1, sizeof(e1)); 
            KLine_RecvFrame(klineresp, 67);
            d_lambda=klineresp[8];      //Lambda
            d_geschw=klineresp[12];     //Geschwindigkeit
            d_drehzahlH=klineresp[13];  //Drehzahl Highbyte
            d_drehzahlL=klineresp[14];  //Drehzahl Lowbyte
            d_lufttemp=klineresp[20];   //Lufttemperatur
            d_motortemp=klineresp[21];  //Motortemperatur
            AddToCols(d_motortemp);     //Mottemp in Buffer bernehmen, wird fr "steigend/sinkend"-Anzeige bentigt
            d_zundwinkel=klineresp[26]; //Zndwinkel
            d_dkwinkel=klineresp[27];   //Drosselklappenwinkel
            d_luftmasseH=klineresp[28]; //Luftmasse Highbyte
            d_luftmasseL=klineresp[29]; //Luftmasse Lowbyte
            d_motmoment=klineresp[30];  //Motormoment
            d_spannung=klineresp[32];   //Batteriespannung
            d_kilometerH=klineresp[44]; //Kilometer Highbyte
            d_kilometerM=klineresp[45]; //Kilometer Midbyte
            d_kilometerL=klineresp[46]; //Kilometer Lowbyte
            d_sekundenH=klineresp[47];  //Sekunden Highbyte
            d_sekundenMH=klineresp[48]; //Sekunden MidHighbyte
            d_sekundenML=klineresp[49]; //Sekunden MidLowbyte
            d_sekundenL=klineresp[50];  //Sekunden Lowbyte
            d_luftdruckH=klineresp[59]; //Luftdruck Highbyte
            d_luftdruckL=klineresp[60]; //Luftdruck Lowbyte
            d_gang=klineresp[61];       //Gang
            if (T0_Overflow) {         //Wenn hier schon ein Timeout vorliegt, keinen weiteren Abruf durchfhren
                SET_BIT(timeout, to_kl_ecua1);
                SET_BIT(timeout, to_kl_ecua2);
                SET_BIT(timeout, to_kl_kombi);
                SET_BIT(timeout, to_kl_ecud);
            } else {
                CLEAR_BIT(timeout, to_kl_ecua1);
                wait55ms();                       //55ms warten vor dem nchsten request (minimale Zeit im "normal timing mode")
                unsigned char e2[] = KL_ECUDATA2;     //Senden des Request fr ECU-AusgangAnalog2 - Response sind 22 Byte
                KLine_SendFrame(e2, sizeof(e2)); 
                KLine_RecvFrame(klineresp, 22);
                d_benzdruckH=klineresp[15]; //Kraftstoffdruck Highbyte
                d_benzdruckL=klineresp[16]; //Kraftstoffdruck Lowbyte
                if (T0_Overflow) {         //Wenn hier schon ein Timeout vorliegt, keinen weiteren Abruf durchfhren
                    SET_BIT(timeout, to_kl_ecua2);
                    SET_BIT(timeout, to_kl_kombi);
                    SET_BIT(timeout, to_kl_ecud);
                } else {
                    CLEAR_BIT(timeout, to_kl_ecua2);
                    wait55ms();                       //55ms warten vor dem nchsten request (minimale Zeit im "normal timing mode")
                    if (klinercv == 0) {          //Nur in jedem 4. Zyklus einholen
                        klinercv = 3;
                        unsigned char k1[] = KL_KOMDATA;           //Senden des Request fr KOMBI-Kraftstoff - Response sind 13 Byte
                        KLine_SendFrame(k1, sizeof(k1)); 
                        KLine_RecvFrame(klineresp, 13);
                        d_kraftstoff=klineresp[5]; 
                        if (T0_Overflow) {         //Wenn hier ein Timeout vorliegt, markieren, dass KOMBI den Timeout erzeugt hat
                            SET_BIT(timeout, to_kl_kombi);
                        } else {
                            CLEAR_BIT(timeout, to_kl_kombi);            
                        }
                    } else {                    //Nur in 3 von 4 Zyklen einholen
                        klinercv--;
                        unsigned char ed[] = KL_ECUDATAD;     //Senden des Request fr ECU-AusgangDigital - Response sind 12 Byte. 
                        KLine_SendFrame(ed, sizeof(ed)); 
                        KLine_RecvFrame(klineresp, 12);
                        d_dig=klineresp[9];        //Aktivitt Motorlfter und Kraftstoffpumpe
                        if (T0_Overflow) {         //Wenn hier ein Timeout vorliegt, markieren, dass ECU-Digital den Timeout erzeugt hat
                            SET_BIT(timeout, to_kl_ecud);
                        } else {
                            CLEAR_BIT(timeout, to_kl_ecud);            
                        }
                    }
                }
            }
            //Wenn bis hier her alle Daten im Zeitlimit eingeholt wurden, sind alle K-Line-Timeout-Bits=0
            //Ansonsten ist das Bit der Einholung, bei der der Timeout eingetreten ist und alle nachfolgenden =1
        }
        //Warte bis Display-Updaterate-Counter berluft
        while (!T0_Overflow);   //Warte bis TMR0 berluft
        TMR0_Reset();
    }
}

//Interrupt Service Routine
void __interrupt(high_priority) ISR(void) {
    //Grosse Routinen am besten in main-Endlosschleife per Flag starten - Abarbeitung des Interrupts hier sorgt evtl dafr, dass Compiler die Routinen duplizieren wrde
    KLine_Enable = 0;       //Groe Verbraucher sofort abschalten, Betrieb erfolgt ab hier nur noch aus Sttzkondensator - noch ca. 200ms bei sparsamem Betrieb
    Kons_Enable = 0;
    if (INTCONbits.INT0F) {                //Shutdown-Erkennung auf INT0
        INTCONbits.INT0F = 0;              //Flag lschen
        EE_WriteByte(0, i_kilometerH);     //Alle Initialwerte in das EEPROM sichern (knnen sich z.B. durch ein gewhltes "Reset" gendert haben. Ohne "Reset bleiben die Werte im EEPROM unverndert)
        EE_WriteByte(1, i_kilometerM);
        EE_WriteByte(2, i_kilometerL);
        EE_WriteByte(3, i_sekundenH);
        EE_WriteByte(4, i_sekundenMH);
        EE_WriteByte(5, i_sekundenML);
        EE_WriteByte(6, i_sekundenL);
        //Nach dem Schreiben toggelt RC1 (Pin 15) so lange bis PIC abschaltet (zum Messen der berlebensdauer bzw. Zeitreserve), 3 Zyklen/1us Pulsdauer
        while (1) {
            Testpin = 1;
            __asm("NOP");
            __asm("NOP");
            __asm("NOP");
            Testpin = 0;
            __asm("NOP");          
        }
    }
}

//Einrichten und Starten des Interrupts
void INT_Init() {
    //Aufsetzen der Interrupts
    TRISCbits.RC0 = 1;          //RC0(INT0)=Input
    ANSELbits.ANS4 = 0;         //RC0(INT0) ist digital   
    INTCONbits.INT0E = 1;       //INT0 enablen (INT0 lst aus bei Shutdown)
    INTCON2bits.INTEDG0 = 0;    //INT0-Interrupt bei falling edge
    INTCONbits.INT0F = 0;       //Evtl. anstehenden Interrupt lschen
    INTCONbits.GIE = 1;         //Interrupts einschalten
}

//Aufsetzen der GPIO Pins
void GPIO_Init() {
    Kons_Enable = 0;            //Display noch ausgeschaltet lassen
    TRISCbits.RC5 = 0;      //RC5(Display Enable)=Output
    TRISCbits.RC4 = 1;      //RC4(Extra-Interface)=Input
    TRISCbits.RC3 = 1;      //RC3(RawSelect)=Input 
    TRISCbits.RC1 = 0;      //RC1(Pin 15)=Output fr Debug-Zwecke toggeln 
}

//Aufsetzen der K-Line Pins
void KLine_Init() {
    KLine_Enable=0;                //K-Line Transceiver noch ausgeschaltet
    KLine_TxPin=1;                //K-Line Transceiver auf Recessive State
    TRISBbits.RB4 = 0;      //RB4(KLine Enable)=Output
    TRISCbits.RC2 = 0;      //RC2(KLine-Tx)=Output
    TRISBbits.RB6 = 1;      //RB6(KLine-Rx)=Input  
    //Transceiver einschalten, danach 2ms warten bis Transceiver ready ist (siehe Datenblatt Power-up to ready ist 1,5ms)
    KLine_Enable=1;
    wait(20);              
}

//Konfiguration des UART fr die Kommunikation mit der Konsole
void UART_Init() {
    TRISBbits.RB5 = 1;      //RB5(Rx)=Input
    TRISBbits.RB7 = 0;      //RB7(Tx)=Output
    ANSELHbits.ANS11 = 0;   //RB5(Rx) ist digital
    TXSTAbits.TX9 = 0;      //8-Bit
    TXSTAbits.TXEN = 1;     //Enable Transmit        
    TXSTAbits.SYNC = 0;     //Async Mode
    TXSTAbits.BRGH = 0;     //Baudrate Low Speed
    RCSTAbits.SPEN = 1;     //Enable Serial Port
    RCSTAbits.RX9 = 0;      //8-Bit
    RCSTAbits.CREN = 1;     //Enable reception bit Rx
    BAUDCONbits.DTRXP = 1;  //Invertierung zu active/idle low , wie vom PC-UART erzeugt wird
    BAUDCONbits.CKTXP = 1;  //Invertierung zu active/idle low, wie vom PC-UART erwartet wird  
    BAUDCONbits.BRG16 = 1;  //16-Bit Baudrate-Gernerierung
    SPBRGH = 0;             //16-Bit Baudzhler: BRG = (12MHz/(9600*16))-1 = 77
    SPBRG = 77; 
}

//Aufsetzen des Timers fr die Display-Update Intervalle
void TMR0_Init() {
    //16-Bit Timer, Prescale 1:32, d.h. berlauf bei 3Mhz alle 0,70 Sekunden
    T0CONbits.TMR0ON = 0;   //Timer noch aus
    T0CONbits.T08BIT = 0;   //16-Bit Timer
    T0CONbits.T0CS = 0;     //Interne 3 Mhz
    T0CONbits.PSA = 0;      //Prescaler assigned
    T0CONbits.T0PS = 4;     //Prescaler 1:32  
    TMR0_Reset();
}

//Zurcksetzen des Timers fr die Display-Update Intervalle
void TMR0_Reset() {
    //16-Bit Timer, Prescale 1:32, d.h. berlauf bei 3Mhz alle 0,70 Sekunden - bei Preset von TMR0H auf 73: berlauf alle 0,50 Sekunden
    T0_Overflow = 0;
    TMR0H = 73;             //TMR0H Preset auf 73: berlauf alle 0,50 Sekunden
    TMR0L = 0;
}

//Liest ein Byte aus dem EEPROM
unsigned char EE_ReadByte(unsigned char adr) {
    EEADR = adr;
    EECON1bits.EEPGD = 0;
    EECON1bits.CFGS = 0;
    EECON1bits.RD = 1; 
    return EEDATA; 
}

//Schreibt ein Byte in das EEPROM
void EE_WriteByte(unsigned char adr, unsigned char dat) {
    EEADR = adr;
    EEDATA = dat;          
    EECON1bits.EEPGD = 0;   
    EECON1bits.CFGS = 0;    
    EECON1bits.WREN = 1; 
    //Interrupt-Disable ist hier nicht ntig, da diese Funktion sowieso nur aus dem Shutdown-Interrupt aufgerufen wird
    EECON2 = 0x55;    
    EECON2 = 0xAA;          
    EECON1bits.WR = 1;  
 //   while(EECON1bits.WR == 1)   //Warten, bis Write komplett ist
    while(PIR2bits.EEIF == 0)   //Warten, bis Write komplett ist
    PIR2bits.EEIF = 0;    
    EECON1bits.WREN = 0;   
}

//Sendet ein Wort bestehend aus dem Control-Byte und dem Daten-Byte ber den UART an die Konsole und wartet evtl auf Response
unsigned char Kons_SendWord(unsigned char cont, unsigned char data) {
    //Control-Byte: Bit(7)=ResponseRequest, Bit(2)=Licht, Bit(1)=RS, Bit(0)=RW (aktuell ist noch kein read-Modus in Konsole implementiert, daher RW immer gleich 0)
    //Wartet dann auf das Response-Byte von der Konsole und empfngt dieses
    unsigned char ret;
    while(!Kons_TxReady); //Warte bis letzter Transmit beendet ist
    TXREG = cont;
    while(!Kons_TxReady); //Warte bis letzter Transmit beendet ist
    TXREG = data;
    if ((cont & 0x80) == 0x80) {          //Cont(7) zeigt an, ob Tastenabfrage stattfinden soll)
        ret=Kons_RecvResp();
    } else {
        ret=0;    
    }
    return ret;
}

//Sendet ein komplettes Display-Update-Frame (16 Zeichen mit LineBreaks etc.) an die Konsole und empfngt am Ende eine Response
unsigned char Kons_SendFrame(unsigned char DChar[16]) {
    unsigned char ct0, ct1;   //Control-Bytes, je nachdem ob mit RS=0 oder RS=1
    unsigned char n, ret;
    
    if (GET_BIT(flags,f_lichtstatus)) {
        ct0 = 0x04;         //Licht=1, RS=0, RW=0
        ct1 = 0x06;         //Licht=1, RS=1, RW=0
    } else {
        ct0 = 0x00;         //Licht=0, RS=0, RW=0
        ct1 = 0x02;         //Licht=0, RS=1, RW=0
    }
    ret=Kons_SendWord(ct0,0x01);     //Kontroll-Befehl Clear Display
    wait(20);                //Wartet 2ms
    for (n=0;n<8;n++) {
        ret=Kons_SendWord(ct1,DChar[n]);
        wait(1);           //Wartet 0,1ms
    }
    ret=Kons_SendWord(ct0,0xC0);     //Kontroll-Befehl NewLine (Adresse auf 0x40)
    wait(1);
     for (n=8;n<15;n++) {
        ret=Kons_SendWord(ct1,DChar[n]);
        wait(1);
    }   
    SET_BIT(ct1,7);         //Response-Request mitsenden (Bit(7)=1)) 
    ret=Kons_SendWord(ct1,DChar[15]);
    wait(1);
    return ret;
}

//Wartet auf und empfngt das Response-Byte von der Konsole
unsigned char Kons_RecvResp() {
    //z.B. nach der Sendung eines Words. Ready-Bytes beim Startup nicht hierber abfragen - Timeout ist da nicht gewnscht.
    unsigned char ret;
    //Debug-Modus (im echten Betrieb auskommentieren)
//    return 3;       //Keine Taste gedrckt
    //Debug-Modus Ende
    ret=RCREG;                  //Evtl schon anstehenden Empfang vom Display lschen (setzt Kons_RxAvailable auf 0)
    while(!Kons_RxAvailable && !T0_Overflow);         //Warte bis Byte empfangen wurde oder Timeout
    if (T0_Overflow) {
        SET_BIT(timeout, to_kons);
        return 3;           //Bei Timeout 3 zurckliefern (= keine Taste gedrckt)
    } else {
        CLEAR_BIT(timeout, to_kons);
        return RCREG;
    }
}

//Warten auf fallende K-Line-Flanke und Empfang eines Datenbytes mit 10400 baud
unsigned char KLine_RecvByte(void) {
	__asm("_a0:  BTFSS	0x81,6,c");	//Zuerst muss KL-Rx=high sein um zu verhindern, dass bereits anstehendes Low (z.B. 25ms Low im Wake-up Event) gleich einen Empfang auslst
	__asm("GOTO     _a0");          //Nein: Weiterwarten    
//	__asm("_a1:  BTFSS	0x81,6,c");	//Startbit (Low=0V) - fallende Flanke auf KL-Rx? PORTB,6(0x81,6) ist KL_Rx
//	__asm("GOTO     _a10");          //Ja: Empfang starten
//  __asm("BTFSS 0xF2,2,c");        //Nein: Timeout (TMR0IF=1)?  0xF2,2 = INTCON,2
//  __asm("GOTO     _a1");          //Nein: Weiterwarten
    while(KLine_RxPin==1 && !T0_Overflow);     //Warte auf fallende Flanke oder Timeout
    if (T0_Overflow) {
        SET_BIT(timeout, to_kline);
        return 0;
    } else {
        __asm("_a10:  CLRF     _dat");
        __asm("MOVLW	8");
        __asm("MOVWF	_n");
        __asm("NOP");
        __asm("NOP");
        __asm("NOP");
        __asm("NOP");
        baud5200();                 //Startbit wurde registriert, warte 1/2 baudzeit (danach gleich noch eine ganze) = 1,5 baud seit der Startbitflanke = Mitte vom ersten bit
        __asm("_a2: NOP");
        baud10400();                //Warte volle baudrate-10 Zyklen (werden in Schleife verbraten), um in naechstes Bit zu kommen
        __asm("BTFSS	0x81,6,c"); //Pruefe Bit PORTB,6(0x81,6) ist KL_Rx
        __asm("GOTO     _a3");		//=0V: (=Logisch 0)
        __asm("BSF		status,0,c");	//=5V: (=Logisch 1) Carry mit 1 besetzen
        __asm("GOTO     _a4");
        __asm("_a3: BCF		status,0,c");	//Carry mit 0 besetzen
        __asm("NOP");				//Damit gleich lang wie BSF-Zweig
        __asm("NOP");
        __asm("_a4: RRCF		_dat,1");	//Carry in Byte verschieben (K-Line:LSB first -> Bits werden von links nach rechts reingeschoben)
        __asm("DECFSZ	_n,1");
        __asm("GOTO     _a2");
        CLEAR_BIT(timeout, to_kline);
        return dat;
    }
}

//Empfngt count Bytes aus dem KLine-Interface und schreibt diese in den bytes-Buffer
void KLine_RecvFrame(unsigned char *bytes, unsigned char count) {
    //Aufzurufen mit z.B. fr 10 erwartete Bytes: "unsigned char frame[10]; KLine_RecvFrame(frame, sizeof(frame));"
    unsigned char x;
    //Debug-Modus (im echten Betrieb auskommentieren)
//    unsigned char y;
//    y = TMR0L;      //Pseudo-Zufallswert
//    for (x=0;x<count;x++) bytes[x]=y++;
    //Debug Modus Ende
    for (x=0;x<count;x++) bytes[x]=KLine_RecvByte();
}

//Sendet ein Byte in das KLine-Interface
void KLine_SendByte(unsigned char byte) {
    dat=byte;
	__asm("MOVLW 	8");
	__asm("MOVWF 	_n");				//8 Bits senden
	__asm("BCF   	0x8b,2,c");			//Startbit setzen (Low = 0V) LATC,2(0x8b,2) ist KL_Tx, zum Schreiben NICHT PORTC,2(0x82,2) verwenden!
	__asm("NOP");
	__asm("NOP");
	__asm("NOP");
	__asm("NOP");
	__asm("NOP");
	baud10400();			//Baudzeit warten
	__asm("_a9: RRCF   	_dat,1");			//Bits durchschieben (K-Line:LSB first -> Bits werden von links nach rechts rausgeschoben)
	__asm("BTFSS 	status,0,c");			//Bit checken = 1?
	__asm("GOTO     _aa");				//nein: Sende 0
	__asm("NOP");						//ja: Sende 1
	__asm("BSF	  	0x8b,2,c");			//High=5V
	__asm("GOTO     _ab");				//Ab zur Pause 
    __asm("_aa: NOP");
	__asm("BCF   	0x8b,2,c");			//Low=0V
	__asm("NOP");
	__asm("NOP");
    __asm("_ab:");
	baud10400();			//Baudzeit warten
	__asm("DECFSZ 	_n,1");				//n=n-1, n=0 (Byte zu Ende)?
	__asm("GOTO  	_a9");			//nein: weiter mit naechstem Bit
	__asm("NOP");						//ja: Stopbit setzen
	__asm("NOP");
	__asm("NOP");
	__asm("NOP");
	__asm("NOP");
	__asm("BSF   	0x8b,2,c");			//Stopbit setzen (High = 5V)
	baud10400();
}

//Sendet einen Frame von beliebiger Anzahl Bytes in das KLine-Interface
void KLine_SendFrame(unsigned char bytes[], unsigned char len) {
    //Es mssen alle zu sendenden Bytes in dem Array vorgegeben sein, incl. das letztes Checksum-Byte (dieses wird nicht ermittelt)
    //Aufrufen immer nur mit "KLIne_SendFrame(frame,sizeof(frame))" - nie mit expliziter Lnge - dies knnte zu Fehlbertragungen in den Bus fhren, wenn len aus versehen nicht genau der Lnge des Frames entspricht
    //sizeof kann leider nicht hier in der Funktion ermittelt werden, weil sizeof hier nur die Lnge des Pointers ermitteln wrde! 
    unsigned char x;
    for (x=0;x<len;x++) KLine_SendByte(bytes[x]);
}

//Generiert die Wakeup-Condition
void KLine_SendWakeup(void) {
    //Wakeup-Condition = (25ms Low, 25ms High)
    KLine_TxPin=0;
    wait(250);
    KLine_TxPin=1;
    wait(250);  
}

//Warteschleife fuer Baudrate 10400
void baud10400(void) {
    //Baudrate 10400 (=96us) bei 12MHz/4 Takt (=288 Zyklen)
    //11 Zyklen werden schon in der Rx-Routine zwischen jedes Bit gesetzt d.h. zusammen mit aufrufendem CALL werden hier 277 Zyklen verbraten
    //2+2+(90*3)+1+2 (CALL + 2*MOV + 90*Schleife + NOP + RETURN)
    __asm("MOVLW  	90");	
	__asm("MOVWF  	_m");
	__asm("_a5: DECFSZ 	_m,1");			
	__asm("GOTO  	_a5");
    __asm("NOP");
}

//Warteschleife fuer 1/2 Baudrate 10400
void baud5200(void) {
    //Eine halbe Baudzeit bei 10400 baud (=48us) bei 12MHz/4 Takt (=144 Zyklen)
    ////2+2+(46*3)+2 (CALL + 2*MOV + 46*Schleife + RETURN)
    __asm("MOVLW  	46");	
	__asm("MOVWF  	_m");
	__asm("_a6: DECFSZ 	_m,1");			
	__asm("GOTO  	_a6");   
}

//Warteschleife fuer cnt * 0,1ms
void wait(unsigned char cnt) {
    //Innere m-Schleife M dauert 3+98*3=297 Zyklen (knapp 0,1ms). Auere n-Schleife wiederholt M je nach cnt
    n=cnt;
    __asm("MOVWF    _n");
    __asm("_a7: MOVLW  	98");	
	__asm("MOVWF  	_m");
    __asm("NOP");
	__asm("_a8: DECFSZ 	_m,1");			
	__asm("GOTO  	_a8");
    __asm("DECFSZ 	_n,1");			
	__asm("GOTO  	_a7");
}

//Warteschleife fuer 55ms
void wait55ms(void) {
    wait(250);           //25ms+25ms+5ms
    wait(250);
    wait(50);
}

//Warteschleife fuer 0,2s
void wait200ms(void) {
    unsigned char n;
    for (n=1;n<=8;n++) wait(250);           //8x 25ms
}

//Ermittelt die Charcter-Codes der jeweiligen Dezimalziffer in value, Rest wird in globale Variable ergebnis geschrieben
unsigned char GetChar(unsigned short value, unsigned short ziffer) {
    //ziffer=10000: 5. Dezimalziffer, ziffer=1000: 4. Dezimalziffer, usw. bis ziffer=1: 1er Dezimalziffer
    //Bercksichtigt dabei auch die Leading-Zero-Elimination, also statt "0010,8" wird "  10,8" ausgegeben
    unsigned char zw;
    if (ziffer == 1) {
        zw = 0x30 + value;     //Zifferncode der 0 ist 0x30, darauffolgende Zahlen jeweils erhht. LZE findet bei nur 1er-Ziffer nicht statt
    } else {
        ergebnis = value % ziffer;  //Rest durch Modulo ermitteln, geht direkt in globale Variable fr zgige Ermittlung der nchsten Ziffer
        zw = value / ziffer;       //Dezimalziffer durch ganzzahlige Division
        if (zw == 0 && GET_BIT(flags,f_lze)) {
            zw = Ch_;              //Bei Ergebnis=0 und eingeschalteter Leading-Zero-Elimination: Leerzeichen statt "0" ausgeben
        } else {
            zw = 0x30 + zw;           //Zifferncode aus Rechenergebnis ermitteln
            CLEAR_BIT(flags,f_lze);
        }
    }
    return zw;
}

//Ermittelt den Charcter-Codes des jeweiligen Gangs
unsigned char getGang(unsigned char gang) {
    //Normalerweise ist es einfach der entsprechende Zifferncode, d.h. z.B. "1" fr gang=1. Nur fr gang=0 soll entsprechend "N" angezeigt werden
    if (gang == 0) {
        return Ch_N;
    } else {
        return gang+0x30;       //Zifferncodes fr "1"..."6" liegen bei 0x31...0x36
    }
}

//Fgt einen Wert zu den Collection-Buffers A bzw. B hinzu
void AddToCols(unsigned char value) {
    //Wird fr die bentigt Anzeige "steigend" bzw. "sinkend" in der Motortemperatur-Anzeige bentigt
    //Buffer A und Buffer B sind jeweil 4 Byte gro.
    //Ein neuer Wert wird solange zum Buffer A hinzugefgt, bis dieser voll ist.
    //Der Durchschnitt des A-Buffers wird dann zum Buffer B hinzugefgt (evtl. ltester Wert wird gelscht), und der A-Buffer wird neu gestartet
    unsigned char n;
    unsigned short sum;
    tcola[pcola] = value;
    if (pcola < 3) {    //Buffer A ist noch nicht voll
        pcola++;
    } else {            //Buffer A ist voll
        sum = 0;        //Buffer A Durchschnitt bilden und diesen in Buffer B eintragen
        for (n=0;n<4;n++) sum = sum + tcola[n];
        sum = sum >> 2;     //sum ist danach zwangslufig auf 1 Byte reduziert 
        pcolb++;            
        if (pcolb == 4) {   //Buffer-Rewind, nach pointer=3 kommt pointer=0
            pcolb = 0;
            SET_BIT(flags, f_colbfull);     //Nach erster vollstndiger Fllung des B-Buffers, full-flag setzen
        }
        tcolb[pcolb] = sum; //pcolb zeigt immer auf aktuellsten Wert, d.h. ltester Wert liegt in pcolb+1 
    }
}

//Liefert nchste ScreenID zurck, abhngig von Tastendrcken
unsigned char setScreenID(unsigned char tasten, unsigned char ohne, unsigned char oben, unsigned char unten, unsigned char beide) {
    //Untersucht den Tastenzustand und liefert die enstprechend gewnschte ScreenID zurck fr ohne Taste, Taste oben, Taste unten oder beide Tasten gedrckt
    //tasten beinhaltet den von der Konsole zurckgeliefrten Tastenzustand (FTOL(3),FTUL(2),FTO(1),FTU(0))
    //Achtung Invertierung! Bit=0 heit "Taste wurde gedrckt" Bit=1 heit "Taste wurde nicht gedrckt"
    unsigned char ret;
    
    switch(tasten & 0x03) {         //Untersucht die Bits FTO und FTU - Achtung Invertierung! Bit=0 heit "Taste wurde gedrckt" Bit=1 heit "Taste wurde nicht gedrckt"
        case 0:         //Beide Tasten wurden gedrckt
                ret = beide;                      
                break;
        case 1:         //Obere Taste wurde gedrckt
                ret = oben;                      
                break;
        case 2:         //Untere Taste wurde gedrckt
                ret = unten;                      
                break;
        case 3:         //Keine Taste wurde gedrckt
                ret = ohne;                      
                break;
    }
    screenIDAlt = screenID;
    return ret;
}

//Erstellt und bermittelt den Screen zur Lichtauswahl, liefert Tastenzustand zurck
unsigned char makeLichtauswahl(void) {
    //"Licht  N"
	//"* Xxx   "
    unsigned char DChar[16];
    DChar[0] = Ch_L;
    DChar[1] = Ch_i;
    DChar[2] = Ch_c;
    DChar[3] = Ch_h;
    DChar[4] = Ch_t;
    DChar[5] = Ch_;
    DChar[6] = Ch_;
    DChar[8] = Ch_Stern;
    DChar[9] = Ch_;
    DChar[13] = Ch_;
    DChar[14] = Ch_;
    DChar[15] = Ch_;
    if (GET_BIT(timeout, to_kl_ecua1)) {
        DChar[7] = Ch_nc;
    } else {
        DChar[7] = getGang(d_gang);
    }
    if (GET_BIT(flags,f_lichtwunsch)) {
        DChar[10] = Ch_E;
        DChar[11] = Ch_i;
        DChar[12] = Ch_n;
    } else {
        DChar[10] = Ch_A;
        DChar[11] = Ch_u;
        DChar[12] = Ch_s;
    }
    return Kons_SendFrame(DChar);
}

//Erstellt und bermittelt den Screen zur Geschwindigkeit, liefert Tastenzustand zurck
unsigned char makeGeschwindigkeit(void) {
    //"Geschw N"
	//"000 km/h"
    unsigned char DChar[16];
    unsigned short geschw;
    geschw = d_geschw * 10;   //Berechnung Anzeigewert=Rohwert*10/8
    geschw = geschw >> 3;
    
    DChar[0] = Ch_G;
    DChar[1] = Ch_e;
    DChar[2] = Ch_s;
    DChar[3] = Ch_c;
    DChar[4] = Ch_h;
    DChar[5] = Ch_w;
    DChar[6] = Ch_;
    DChar[11] = Ch_;
    DChar[12] = Ch_k;
    DChar[13] = Ch_m;
    DChar[14] = Ch_Quer;
    DChar[15] = Ch_h;  
    if (GET_BIT(timeout, to_kl_ecua1)) {
        DChar[7] = Ch_nc;
        DChar[8] = Ch_nc;
        DChar[9] = Ch_nc;
        DChar[10] = Ch_nc;
    } else {
        DChar[7] = getGang(d_gang);
        SET_BIT(flags,f_lze);
        DChar[8] = GetChar(geschw,100);
        DChar[9] = GetChar(ergebnis,10);
        DChar[10] = GetChar(ergebnis,1);
    }
    return Kons_SendFrame(DChar);  
}

//Erstellt und bermittelt den Screen zur Drehzahl numerischen Anzeige, liefert Tastenzustand zurck
unsigned char makeDrehzahlNum(void) {
    //"Drehz  N"
	//"0000 U/m"
    unsigned char DChar[16];
    unsigned short drehzahl;
    //16Bit-Variable aus den 2 einzelnen Bytes generieren - Compiler macht das sehr umstndlich
    __asm("movff	_d_drehzahlL,makeDrehzahlNum@drehzahl");
    __asm("movff	_d_drehzahlH,makeDrehzahlNum@drehzahl+1"); 
    drehzahl = drehzahl >> 2;       //Berechnung Anzeigewert=Rohwert/4
    
    DChar[0] = Ch_D;
    DChar[1] = Ch_r;
    DChar[2] = Ch_e;
    DChar[3] = Ch_h;
    DChar[4] = Ch_z;
    DChar[5] = Ch_;
    DChar[6] = Ch_;
    DChar[12] = Ch_;
    DChar[13] = Ch_U;
    DChar[14] = Ch_Quer;
    DChar[15] = Ch_m;   
    if (GET_BIT(timeout, to_kl_ecua1)) {
        DChar[7] = Ch_nc;
        DChar[8] = Ch_nc;
        DChar[9] = Ch_nc;
        DChar[10] = Ch_nc;
        DChar[11] = Ch_nc;
    } else {
        DChar[7] = getGang(d_gang);
        SET_BIT(flags,f_lze);
        DChar[8] = GetChar(drehzahl,1000);
        DChar[9] = GetChar(ergebnis,100);
        DChar[10] = GetChar(ergebnis,10);
        DChar[11] = GetChar(ergebnis,1);
    }
    return Kons_SendFrame(DChar);      
}

//Erstellt und bermittelt den Screen zur Drehzahl-Balkenanzeige, liefert Tastenzustand zurck
unsigned char makeDrehzahlBar(void) {
    //"Drehz  N"
	//"XXXXXXXX"
    unsigned char DChar[16];
    unsigned short drehzahl;
    unsigned char anzpix;
    unsigned char x;
    //16Bit-Variable aus den 2 einzelnen Bytes generieren - Compiler macht das sehr umstndlich
    __asm("movff	_d_drehzahlL,makeDrehzahlBar@drehzahl");
    __asm("movff	_d_drehzahlH,makeDrehzahlBar@drehzahl+1"); 
    drehzahl = drehzahl >> 2;       //Berechnung Anzeigewert=Rohwert/4
    
    DChar[0] = Ch_D;
    DChar[1] = Ch_r;
    DChar[2] = Ch_e;
    DChar[3] = Ch_h;
    DChar[4] = Ch_z;
    DChar[5] = Ch_;
    DChar[6] = Ch_;
    DChar[8] = Ch_;     //Letzten Bargraph vorest ganz lschen
    DChar[9] = Ch_;
    DChar[10] = Ch_;
    DChar[11] = Ch_;
    DChar[12] = Ch_;
    DChar[13] = Ch_;
    DChar[14] = Ch_;
    DChar[15] = Ch_;      
    if (GET_BIT(timeout, to_kl_ecua1)) {
        DChar[7] = Ch_nc;
        DChar[8] = Ch_nc;     //Bargraph nur mit Minus fllen
        DChar[9] = Ch_nc;
        DChar[10] = Ch_nc;
        DChar[11] = Ch_nc;
        DChar[12] = Ch_nc;
        DChar[13] = Ch_nc;
        DChar[14] = Ch_nc;
        DChar[15] = Ch_nc;
    } else {
        DChar[7] = getGang(d_gang);
        anzpix = drehzahl / 167;        //Angezeigt werden soll 1000U/min pro Character. Ein Character hat 5 Anzeigepixel + 1 Zwischenraumpixel Breite, d.h. 1000/6 = ca. 167
        x = 8;
        while (anzpix >= 6) {           //Solange volle (5/5) Character anzeigen bis anzpix < 6 wird
            DChar[x] = Ch_Bar5;
            x++;
            anzpix = anzpix -6;
        }
        if (anzpix > 0) DChar[x] = 0x02 + anzpix; //Letzten Character errechnen. Wenn anzpix=0 ist, nichts anzeigen - restliche Elemente wurden bereits vorab gelscht 
    }
    return Kons_SendFrame(DChar);      
}

//Erstellt und bermittelt den Screen zur Fahrstrecke, liefert Tastenzustand zurck
unsigned char makeDistanz(void) {
    //"Dist   N"
	//" 00000km"
    unsigned char DChar[16];
    __uint24 distanz;
    __uint24 akilometer;
    __uint24 ikilometer;
    
    //24Bit-Variable aus den 3 einzelnen Bytes generieren - Compiler macht das sehr umstndlich
    akilometer = TMR0L;      //Dummy - Das ist leider ntig, sonst optimiert der Compiler kilometer weg, weil er nicht in den Assembly-Code schaut
    __asm("movff	_d_kilometerL,makeDistanz@akilometer");
    __asm("movff	_d_kilometerM,makeDistanz@akilometer+1");
    __asm("movff	_d_kilometerH,makeDistanz@akilometer+2");     
    //selbers fr Inital-Stand
    ikilometer = TMR0L; 
    __asm("movff	_i_kilometerL,makeDistanz@ikilometer");
    __asm("movff	_i_kilometerM,makeDistanz@ikilometer+1");
    __asm("movff	_i_kilometerH,makeDistanz@ikilometer+2"); 
    //Aktuellen Gesamt-km-Stand in Fahrdistanz umrechnen
    distanz = akilometer - ikilometer;
    
    DChar[0] = Ch_D;
    DChar[1] = Ch_i;
    DChar[2] = Ch_s;
    DChar[3] = Ch_t;
    DChar[4] = Ch_;
    DChar[5] = Ch_;
    DChar[6] = Ch_;
    DChar[8] = Ch_;
    DChar[14] = Ch_k;
    DChar[15] = Ch_m;
    if (GET_BIT(timeout, to_kl_ecua1)) {
        DChar[7] = Ch_nc;
        DChar[9] = Ch_nc;
        DChar[10] = Ch_nc;
        DChar[11] = Ch_nc;
        DChar[12] = Ch_nc;
        DChar[13] = Ch_nc;
    } else {
        DChar[7] = getGang(d_gang);
        SET_BIT(flags,f_lze);
        DChar[9] = GetChar(distanz,10000);
        DChar[10] = GetChar(ergebnis,1000);
        DChar[11] = GetChar(ergebnis,100);
        DChar[12] = GetChar(ergebnis,10);
        CLEAR_BIT(flags,f_lze);
        DChar[13] = GetChar(ergebnis,1);
    }
    return Kons_SendFrame(DChar);      
}

//Erstellt und bermittelt den Screen zur Fahrzeit, liefert Tastenzustand zurck
unsigned char makeFahrzeit(void) {
    //"Brt.Zt N"
	//"  000:00"
    unsigned char DChar[16];
    unsigned long asekunden;
    unsigned long isekunden;
    unsigned long gsekunden;
    unsigned long gminuten;
    unsigned int stunden;
    unsigned char minuten;
    
    //32Bit-Variable aus den 4 einzelnen Bytes generieren - Compiler macht das sehr umstndlich
    asekunden = TMR0L;      //Dummy - Das ist leider ntig, sonst optimiert der Compiler gminuten weg, weil er nicht in den Assembly-Code schaut
    __asm("movff	_d_sekundenL,makeFahrzeit@asekunden");
    __asm("movff	_d_sekundenML,makeFahrzeit@asekunden+1");
    __asm("movff	_d_sekundenMH,makeFahrzeit@asekunden+2");
    __asm("movff	_d_sekundenH,makeFahrzeit@asekunden+3");
    //Selbes fr Initial-Werte
    isekunden = TMR0L;   
    __asm("movff	_i_sekundenL,makeFahrzeit@isekunden");
    __asm("movff	_i_sekundenML,makeFahrzeit@isekunden+1");
    __asm("movff	_i_sekundenMH,makeFahrzeit@isekunden+2");
    __asm("movff	_i_sekundenH,makeFahrzeit@isekunden+3");
   //Gesamt-Minuten errechnen und in Stunden und Rest-Minuten umrechnen
    gsekunden = asekunden - isekunden;
    gminuten = gsekunden / 60;
    stunden = gminuten / 60;
    minuten = gminuten % 60;
    
    DChar[0] = Ch_B;
    DChar[1] = Ch_r;
    DChar[2] = Ch_t;
    DChar[3] = Ch_Punkt;
    DChar[4] = Ch_Z;
    DChar[5] = Ch_t;
    DChar[6] = Ch_;
    DChar[8] = Ch_;
    DChar[9] = Ch_;
    DChar[13] = Ch_DpPkt;  
    if (GET_BIT(timeout, to_kl_ecua1)) {
        DChar[7] = Ch_nc;
        DChar[10] = Ch_nc;
        DChar[11] = Ch_nc;
        DChar[12] = Ch_nc;
        DChar[14] = Ch_nc;
        DChar[15] = Ch_nc;
    } else {
        DChar[7] = getGang(d_gang);
        SET_BIT(flags,f_lze);
        DChar[10] = GetChar(stunden,100);
        DChar[11] = GetChar(ergebnis,10);
        CLEAR_BIT(flags,f_lze);
        DChar[12] = GetChar(ergebnis,1);
        DChar[14] = GetChar(minuten,10);
        DChar[15] = GetChar(ergebnis,1);
    }
    return Kons_SendFrame(DChar);      
}

//Erstellt und bermittelt den Screen zur Durchschnittsgeschwindigkeit, liefert Tastenzustand zurck
unsigned char makeVDurchschnitt(void) {
    //"OB.Ges N"
	//"000 km/h"
    unsigned char DChar[16];
    unsigned char geschw;
    __uint24 distanz;
    __uint24 akilometer;
    __uint24 ikilometer; 
    unsigned long asekunden;
    unsigned long isekunden;
    unsigned long gsekunden;

    //24Bit-Variable aus den 3 einzelnen Bytes generieren - Compiler macht das sehr umstndlich
    akilometer = TMR0L;      //Dummy - Das ist leider ntig, sonst optimiert der Compiler kilometer weg, weil er nicht in den Assembly-Code schaut
    __asm("movff	_d_kilometerL,makeVDurchschnitt@akilometer");
    __asm("movff	_d_kilometerM,makeVDurchschnitt@akilometer+1");
    __asm("movff	_d_kilometerH,makeVDurchschnitt@akilometer+2");     
    //selbers fr Inital-Stand
    ikilometer = TMR0L; 
    __asm("movff	_i_kilometerL,makeVDurchschnitt@ikilometer");
    __asm("movff	_i_kilometerM,makeVDurchschnitt@ikilometer+1");
    __asm("movff	_i_kilometerH,makeVDurchschnitt@ikilometer+2"); 
    //Aktuellen Gesamt-km-Stand in Fahrdistanz umrechnen
    distanz = akilometer - ikilometer;  
    
    //32Bit-Variable aus den 4 einzelnen Bytes generieren - Compiler macht das sehr umstndlich
    asekunden = TMR0L;      //Dummy - Das ist leider ntig, sonst optimiert der Compiler gminuten weg, weil er nicht in den Assembly-Code schaut
    __asm("movff	_d_sekundenL,makeVDurchschnitt@asekunden");
    __asm("movff	_d_sekundenML,makeVDurchschnitt@asekunden+1");
    __asm("movff	_d_sekundenMH,makeVDurchschnitt@asekunden+2");
    __asm("movff	_d_sekundenH,makeVDurchschnitt@asekunden+3");
    //Selbes fr Initial-Werte
    isekunden = TMR0L;   
    __asm("movff	_i_sekundenL,makeVDurchschnitt@isekunden");
    __asm("movff	_i_sekundenML,makeVDurchschnitt@isekunden+1");
    __asm("movff	_i_sekundenMH,makeVDurchschnitt@isekunden+2");
    __asm("movff	_i_sekundenH,makeVDurchschnitt@isekunden+3");
   //Gesamt-Minuten errechnen und in Stunden und Rest-Minuten umrechnen
    gsekunden = asekunden - isekunden;
    distanz = distanz * 3600;         //Berechnung: geschw = distanz/gsekunden*3600
    geschw = distanz / gsekunden;
    
    DChar[0] = Ch_DSchnt;
    DChar[1] = Ch_B;
    DChar[2] = Ch_Punkt;
    DChar[3] = Ch_G;
    DChar[4] = Ch_e;
    DChar[5] = Ch_s;
    DChar[6] = Ch_;
    DChar[11] = Ch_;
    DChar[12] = Ch_k;
    DChar[13] = Ch_m;
    DChar[14] = Ch_Quer;
    DChar[15] = Ch_h; 
    if (GET_BIT(timeout, to_kl_ecua1)) {
        DChar[7] = Ch_nc;
        DChar[8] = Ch_nc;
        DChar[9] = Ch_nc;
        DChar[10] = Ch_nc;
    } else {
        DChar[7] = getGang(d_gang);
        SET_BIT(flags,f_lze);
        DChar[8] = GetChar(geschw,100);
        DChar[9] = GetChar(ergebnis,10);
        DChar[10] = GetChar(ergebnis,1);
    }
    return Kons_SendFrame(DChar);    
}

//Erstellt und bermittelt den Screen zur Reset-Auswahl, liefert Tastenzustand zurck
unsigned char makeReset(void) {
    //"J Reset?"
	//"N       "
    unsigned char DChar[16];
    DChar[0] = Ch_J;
    DChar[1] = Ch_;
    DChar[2] = Ch_R;
    DChar[3] = Ch_e;
    DChar[4] = Ch_s;
    DChar[5] = Ch_e;
    DChar[6] = Ch_t;
    DChar[7] = Ch_Frage;
    DChar[8] = Ch_N;
    DChar[9] = Ch_;
    DChar[10] = Ch_;
    DChar[11] = Ch_;
    DChar[12] = Ch_;
    DChar[13] = Ch_;
    DChar[14] = Ch_;
    DChar[15] = Ch_;
    return Kons_SendFrame(DChar);      
}

//Erstellt und bermittelt den Screen zur Wassertemperatur, liefert Tastenzustand zurck
unsigned char makeKuehltemp(void) {
    //"Mottmp N"
	//"x 000C^"
    unsigned char DChar[16];
    unsigned char palt;
    unsigned short temp;
    if (d_motortemp<64) {temp=64-d_motortemp;} else {temp=d_motortemp-64;}  //Berechnung Anzeigewert=(Rohwert-64)*3/4
    temp = temp * 3;
    temp = temp >> 2;
            
    DChar[0] = Ch_M;
    DChar[1] = Ch_o;
    DChar[2] = Ch_t;
    DChar[3] = Ch_t;
    DChar[4] = Ch_m;
    DChar[5] = Ch_p;
    DChar[6] = Ch_;
    DChar[9] = Ch_;
    DChar[13] = Ch_Grad;
    DChar[14] = Ch_C;
    if (GET_BIT(timeout, to_kl_ecud)) {
        DChar[8] = Ch_nc;
    } else {
        if (GET_BIT(d_dig, d_luefter)) {        //Wenn der Lfter luft 
            if (GET_BIT(flags, f_prop)) {       //Abwechselnd Propeller in Stellung 1 und Stellung 2 anzeigen
                DChar[8] = Ch_prop1;
            } else {
                DChar[8] = Ch_prop2;
            }
            TOGGLE_BIT(flags, f_prop);
        } else {
            DChar[8] = Ch_;
        }
    }
    if (GET_BIT(timeout, to_kl_ecua1)) {
        DChar[7] = Ch_nc;
        DChar[10] = Ch_nc;
        DChar[11] = Ch_nc;
        DChar[12] = Ch_nc;
        DChar[15] = Ch_nc;
    } else {
        DChar[7] = getGang(d_gang);
        SET_BIT(flags,f_lze);
        if (d_motortemp<64) {DChar[10] = Ch_Minus; ergebnis=temp;} else {DChar[10] = GetChar(temp,100);}
        DChar[11] = GetChar(ergebnis,10);
        DChar[12] = GetChar(ergebnis,1);
        //Ab hier Errechnung und Anzeige des "steigend/sinkend"-Zeichens
        if (GET_BIT(flags, f_colbfull)) {       //Anzeige erst nachdem der Buffer voll gefllt ist, also nach ca. 8 Sekunden (bei Dateneinholung alle 0,5s) - ansonsten wird Mll angezeigt
            if (pcolb == 3) {       //Pointer auf ltesten Wert im B-Buffer ermitteln)
                palt = 0;
            } else {
                palt = pcolb + 1;
            }
            //Bei einer Update-Rate der Werte von ca. 0,5sek (Einholung der Daten ber K-Line), werden also die Temp-Daten der letzten 4*4*0,5 = 8 Sekunden vorgehalten, verpackt in 4 Durchschnittswerte von je 4 Einzelwerten
            //Im Folgenden wird der Durchschnitt der 4 ltesten Einzelwerte (also Werte von vor 6-8sek) mit dem Durchschnitt der 4 aktuellsten Einzelwerte (also Werte von vor 0-2sek) verglichen
            if (tcolb[pcolb] >= (tcolb[palt]+4)) {           //Wenn aktuellster Wert um mindestens 4 Rohwerte (=3C) groesser ist als ltester Wert: Kriterium fr "Steigend"
                DChar[15] = Ch_PfeilO;
            } else if (tcolb[pcolb] <= (tcolb[palt]-4)) {    //Wenn aktuellster Wert um mindestens 4 Rohwerte (=3C) kleiner ist als ltester Wert: Kriterium fr "Sinkend"
                DChar[15] = Ch_PfeilU;
            } else {                                        // Wenn aktuellster Wert innerhalb +4/-4 Rohwerte zum ltesten Wert liegt: Kriterium fr "Gleichbleibend"
                DChar[15] = Ch_;
            }
        }
    }
    return Kons_SendFrame(DChar);      
}

//Erstellt und bermittelt den Screen zur Auentemperatur, liefert Tastenzustand zurck
unsigned char makeAussentemp(void) {
    //"Lfttmp N"
	//"  000C "
    unsigned char DChar[16];
    unsigned short temp;
    if (d_lufttemp<64) {temp=64-d_lufttemp;} else {temp=d_lufttemp-64;}  //Berechnung Anzeigewert=(Rohwert-64)*3/4
    temp = temp * 3;
    temp = temp >> 2;
    
    DChar[0] = Ch_L;
    DChar[1] = Ch_f;
    DChar[2] = Ch_t;
    DChar[3] = Ch_t;
    DChar[4] = Ch_m;
    DChar[5] = Ch_p;
    DChar[6] = Ch_;
    DChar[8] = Ch_;
    DChar[9] = Ch_;
    DChar[13] = Ch_Grad;
    DChar[14] = Ch_C;
    DChar[15] = Ch_; 
    if (GET_BIT(timeout, to_kl_ecua1)) {
        DChar[7] = Ch_nc;
        DChar[10] = Ch_nc;
        DChar[11] = Ch_nc;
        DChar[12] = Ch_nc;
    } else {
        DChar[7] = getGang(d_gang);
        SET_BIT(flags,f_lze);
        if (d_lufttemp<64) {DChar[10] = Ch_Minus; ergebnis=temp;} else {DChar[10] = GetChar(temp,100);}
        DChar[11] = GetChar(ergebnis,10);
        DChar[12] = GetChar(ergebnis,1);
    }
    return Kons_SendFrame(DChar);      
}

//Erstellt und bermittelt den Screen zum Luftdruck, liefert Tastenzustand zurck
unsigned char makeLuftdruck(void) {
    //"Lftdrk N"
	//"0000 hPa"
    unsigned char DChar[16];
    unsigned short druck;
    unsigned char tmp;
    tmp = TMR0L;      //Dummy - Das ist leider ntig, sonst optimiert der Compiler tmp weg, weil er nicht in den Assembly-Code schaut
    druck = d_luftdruckL * 10;      //Berechnung Anzeigewert = Rohwert*10/256 = RohH*10 + RohL*10/256 (um Skalierung in 24 Bit zu vermeiden) 
    __asm("movff	makeLuftdruck@druck+1,makeLuftdruck@tmp");        //tmp = druck/256; 
    druck = d_luftdruckH * 10;
    druck = druck + tmp;
    
    DChar[0] = Ch_L;
    DChar[1] = Ch_f;
    DChar[2] = Ch_t;
    DChar[3] = Ch_d;
    DChar[4] = Ch_r;
    DChar[5] = Ch_k;
    DChar[6] = Ch_;
    DChar[12] = Ch_;
    DChar[13] = Ch_h;
    DChar[14] = Ch_P;
    DChar[15] = Ch_a;
    if (GET_BIT(timeout, to_kl_ecua1)) {
        DChar[7] = Ch_nc;
        DChar[8] = Ch_nc;
        DChar[9] = Ch_nc;
        DChar[10] = Ch_nc;
        DChar[11] = Ch_nc;
    } else {
        DChar[7] = getGang(d_gang);
        SET_BIT(flags,f_lze);
        DChar[8] = GetChar(druck,1000);
        DChar[9] = GetChar(ergebnis,100);
        DChar[10] = GetChar(ergebnis,10);
        DChar[11] = GetChar(ergebnis,1);
    }
    return Kons_SendFrame(DChar);      
}

//Erstellt und bermittelt den Screen zur Bordspannung, liefert Tastenzustand zurck
unsigned char makeSpannung(void) {
    //"V-Batt N"
	//"  00,0 V"
    unsigned char DChar[16];
    unsigned char spannung;
    unsigned short tmp;
    spannung = TMR0L;      //Dummy - Das ist leider ntig, sonst optimiert der Compiler spannung weg, weil er nicht in den Assembly-Code schaut
    tmp = d_spannung * 241;   //Berechnung Anzeigewert=Rohwert*241/256
    __asm("movff	makeSpannung@tmp+1,makeSpannung@spannung");    //spannung = tmp/256;
    DChar[0] = tmp;     //Dummy - Das ist leider ntig, sonst optimiert der Compiler tmp weg, weil er nicht in den Assembly-Code schaut
    
    DChar[0] = Ch_V;
    DChar[1] = Ch_Minus;
    DChar[2] = Ch_B;
    DChar[3] = Ch_a;
    DChar[4] = Ch_t;
    DChar[5] = Ch_t;
    DChar[6] = Ch_;
    DChar[8] = Ch_;
    DChar[9] = Ch_;
    DChar[12] = Ch_Komma;
    DChar[14] = Ch_;
    DChar[15] = Ch_V;
    if (GET_BIT(timeout, to_kl_ecua1)) {
        DChar[7] = Ch_nc;
        DChar[10] = Ch_nc;
        DChar[11] = Ch_nc;
        DChar[13] = Ch_nc;
    } else {
        DChar[7] = getGang(d_gang);
        SET_BIT(flags,f_lze);
        DChar[10] = GetChar(spannung,100);
        CLEAR_BIT(flags,f_lze);
        DChar[11] = GetChar(ergebnis,10);
        DChar[13] = GetChar(ergebnis,1);
    }
    return Kons_SendFrame(DChar);      
}

//Erstellt und bermittelt den Screen zur Kraftstoffmenge, liefert Tastenzustand zurck
unsigned char makeKraftstoff(void) {
    //"Benzin N"
	//"x 00 l  "
    unsigned char DChar[16];
    unsigned char liter;
    liter = d_kraftstoff;
    
    DChar[0] = Ch_B;
    DChar[1] = Ch_e;
    DChar[2] = Ch_n;
    DChar[3] = Ch_z;
    DChar[4] = Ch_i;
    DChar[5] = Ch_n;
    DChar[6] = Ch_;
    DChar[9] = Ch_;
    DChar[12] = Ch_;
    DChar[13] = Ch_l;
    DChar[14] = Ch_;
    DChar[15] = Ch_;    
    if (GET_BIT(timeout, to_kl_ecua1)) {
        DChar[7] = Ch_nc;
    } else {
        DChar[7] = getGang(d_gang);
    }   
    if (GET_BIT(timeout, to_kl_ecud)) {
        DChar[8] = Ch_nc;
    } else {
        if (GET_BIT(d_dig, d_bpumpe)) {         //Wenn die Pumpe luft 
            if (GET_BIT(flags, f_prop)) {       //Abwechselnd Propeller in Stellung 1 und Stellung 2 anzeigen
                DChar[8] = Ch_prop1;
            } else {
                DChar[8] = Ch_prop2;
            }
            TOGGLE_BIT(flags, f_prop);
        } else {
            DChar[8] = Ch_;
        }
    }
    if (GET_BIT(timeout, to_kl_kombi)) {
        DChar[10] = Ch_nc;
        DChar[11] = Ch_nc;
    } else {
        SET_BIT(flags,f_lze);
        DChar[10] = GetChar(liter,10);
        DChar[11] = GetChar(ergebnis,1);
    }
    return Kons_SendFrame(DChar);      
}

//Erstellt und bermittelt den Screen zum Einspritzdruck, liefert Tastenzustand zurck
unsigned char makeSpritzdruck(void) {
    //"B-Drck N"
	//" 0,0 bar"
    unsigned char DChar[16];
    unsigned short druck;
    //16Bit-Variable aus den 2 einzelnen Bytes generieren - Compiler macht das sehr umstndlich
    __asm("movff	_d_benzdruckL,makeSpritzdruck@druck");
    __asm("movff	_d_benzdruckH,makeSpritzdruck@druck+1"); 
    druck = druck / 500;        //Berechnung Anzeigewert=Rohwert/500 (eigentlich nur /5 aber hier soll bar statt hPa angezeigt werden)
    
    DChar[0] = Ch_B;
    DChar[1] = Ch_Minus;
    DChar[2] = Ch_D;
    DChar[3] = Ch_r;
    DChar[4] = Ch_c;
    DChar[5] = Ch_k;
    DChar[6] = Ch_;
    DChar[8] = Ch_;
    DChar[10] = Ch_Komma;
    DChar[12] = Ch_;
    DChar[13] = Ch_b;
    DChar[14] = Ch_a;
    DChar[15] = Ch_r;   
    if (GET_BIT(timeout, to_kl_ecua1)) {
        DChar[7] = Ch_nc;
    } else {
        DChar[7] = getGang(d_gang);
    }
    if (GET_BIT(timeout, to_kl_ecua2)) {
        DChar[9] = Ch_nc;
        DChar[11] = Ch_nc;
    } else {
        DChar[9] = GetChar(druck,10);
        DChar[11] = GetChar(ergebnis,1);
    }
    return Kons_SendFrame(DChar);      
}

//Erstellt und bermittelt den Screen zur Zuendwinkel, liefert Tastenzustand zurck
unsigned char makeZuendwinkel(void) {
    //"Znd-a N"
	//" -00   "
    unsigned char DChar[16];
    unsigned short winkel;
    if (d_zundwinkel<128) {winkel=128-d_zundwinkel;} else {winkel=d_zundwinkel-128;}  //Berechnung Anzeigewert=(Rohwert-128)*3/4
    winkel = winkel * 3;
    winkel = winkel >> 2;
    
    DChar[0] = Ch_Z;
    DChar[1] = Ch_ue;
    DChar[2] = Ch_n;
    DChar[3] = Ch_d;
    DChar[4] = Ch_Minus;
    DChar[5] = Ch_Alpha;
    DChar[6] = Ch_;
    DChar[8] = Ch_;
    if (d_zundwinkel<128) {DChar[9] = Ch_Minus;} else {DChar[9] = Ch_;}
    DChar[12] = Ch_;
    DChar[13] = Ch_Grad;
    DChar[14] = Ch_;
    DChar[15] = Ch_;
    if (GET_BIT(timeout, to_kl_ecua1)) {
        DChar[7] = Ch_nc;
        DChar[10] = Ch_nc;
        DChar[11] = Ch_nc;
    } else {
        DChar[7] = getGang(d_gang);
        SET_BIT(flags,f_lze);
        DChar[10] = GetChar(winkel,10);
        DChar[11] = GetChar(ergebnis,1);
    }
    return Kons_SendFrame(DChar);      
}

//Erstellt und bermittelt den Screen zum Drosselklappenwinkel, liefert Tastenzustand zurck
unsigned char makeDrosselklappe(void) {
    //"DKlppe N"
	//"  000 % "
    unsigned char DChar[16];
    unsigned char prozent;
    unsigned short tmp;
    tmp = d_dkwinkel * 201;     //Berechnung Anzeigewert=Rohwert*201/512
    __asm("movff	makeDrosselklappe@tmp+1,makeDrosselklappe@prozent");        //prozent = tmp/256  
    prozent = prozent >> 1;     //prozent=prozent/2 , also insgesamt nun /512
    DChar[0] = tmp;     //Dummy - Das ist leider ntig, sonst optimiert der Compiler tmp weg, weil er nicht in den Assembly-Code schaut

    DChar[0] = Ch_D;
    DChar[1] = Ch_K;
    DChar[2] = Ch_l;
    DChar[3] = Ch_p;
    DChar[4] = Ch_p;
    DChar[5] = Ch_e;
    DChar[6] = Ch_;
    DChar[8] = Ch_;
    DChar[9] = Ch_;
    DChar[13] = Ch_;
    DChar[14] = Ch_Proz;
    DChar[15] = Ch_;
    if (GET_BIT(timeout, to_kl_ecua1)) {
        DChar[7] = Ch_nc;
        DChar[10] = Ch_nc;
        DChar[11] = Ch_nc;
        DChar[12] = Ch_nc;
    } else {
        DChar[7] = getGang(d_gang);
        SET_BIT(flags,f_lze);
        DChar[10] = GetChar(prozent,100);
        DChar[11] = GetChar(ergebnis,10);
        DChar[12] = GetChar(ergebnis,1);
    }
    return Kons_SendFrame(DChar);      
}

//Erstellt und bermittelt den Screen zur Drehmomentanzeige, liefert Tastenzustand zurck
unsigned char makeDrehmoment(void) {
    //"DMomnt N"
	//"  00 Nm "
    //"  000 % "
    unsigned char DChar[16];
    unsigned char erg;
    unsigned short tmp;
    erg = TMR0L;      //Dummy - Das ist leider ntig, sonst optimiert der Compiler erg weg, weil er nicht in den Assembly-Code schaut
    tmp = d_motmoment * 101;     //Berechnung Prozent: Anzeigewert=Rohwert*101/256
    //tmp = d_motmoment * 83;      //Berechnung Nm: Anzeigewert=Rohwert*83/256
    __asm("movff	makeDrehmoment@tmp+1,makeDrehmoment@erg");  //erg = tmp/256    
    DChar[0] = tmp;     //Dummy - Das ist leider ntig, sonst optimiert der Compiler tmp weg, weil er nicht in den Assembly-Code schaut
            
    DChar[0] = Ch_D;
    DChar[1] = Ch_M;
    DChar[2] = Ch_o;
    DChar[3] = Ch_m;
    DChar[4] = Ch_n;
    DChar[5] = Ch_t;
    DChar[6] = Ch_;
    DChar[8] = Ch_;
    DChar[9] = Ch_;
    //DChar[12] = Ch_;
    //DChar[13] = Ch_N;
    //DChar[14] = Ch_m;
    DChar[13] = Ch_;
    DChar[14] = Ch_Proz;
    DChar[15] = Ch_;
    if (GET_BIT(timeout, to_kl_ecua1)) {
        DChar[7] = Ch_nc;
        DChar[10] = Ch_nc;
        DChar[11] = Ch_nc;
        DChar[12] = Ch_nc;
    } else {
        DChar[7] = getGang(d_gang);
        SET_BIT(flags,f_lze);
        //DChar[10] = GetChar(erg,10);
        //DChar[11] = GetChar(ergebnis,1);
        DChar[10] = GetChar(erg,100);
        DChar[11] = GetChar(ergebnis,10);
        DChar[12] = GetChar(ergebnis,1);
    }
    return Kons_SendFrame(DChar);      
}

//Erstellt und bermittelt den Screen zur Leistungs Anzeige, liefert Tastenzustand zurck
unsigned char makeLeistung(void) {
    //"Leistng N"
	//"  00 PS "
    unsigned char DChar[16];
    unsigned char leistung;
    unsigned short drehroh;
    __uint24 zwert;
    drehroh = TMR0L;      //Dummy - Das ist leider ntig, sonst optimiert der Compiler die Variable weg, weil er nicht in den Assembly-Code schaut
    //16Bit-Variable aus den 2 einzelnen Bytes generieren - Compiler macht das sehr umstndlich
    __asm("movff	_d_drehzahlL,makeLeistung@drehroh");
    __asm("movff	_d_drehzahlH,makeLeistung@drehroh+1"); 
    zwert = drehroh * d_motmoment;       //Berechnung Anzeigewert = DrehzahlRohwert * MotormomentRohwert / 86651
    leistung = zwert / 86651;           //Der Divisor 86651 ist die Kombination aller Umrechnungsparameter (Rohwerte zu Absolutwerten, U/min zu U/sek, KW zu PS, 2*PI)
    
    DChar[0] = Ch_L;
    DChar[1] = Ch_e;
    DChar[2] = Ch_i;
    DChar[3] = Ch_s;
    DChar[4] = Ch_t;
    DChar[5] = Ch_n;
    DChar[6] = Ch_g;
    DChar[8] = Ch_;
    DChar[9] = Ch_;
    DChar[12] = Ch_;
    DChar[13] = Ch_P;
    DChar[14] = Ch_S;
    DChar[15] = Ch_;   
    if (GET_BIT(timeout, to_kl_ecua1)) {
        DChar[7] = Ch_nc;
        DChar[10] = Ch_nc;
        DChar[11] = Ch_nc;
    } else {
        DChar[7] = getGang(d_gang);
        SET_BIT(flags,f_lze);
        DChar[10] = GetChar(leistung,10);
        DChar[11] = GetChar(ergebnis,1);
    }
    return Kons_SendFrame(DChar);      
}

//Erstellt und bermittelt den Screen zur Luftmassenanzeige, liefert Tastenzustand zurck
unsigned char makeLuftmasse(void) {
    //"LMasse N"
	//"0000kg/h"
    unsigned char DChar[16];    
    unsigned short masse;

    //16Bit-Variable aus den 2 einzelnen Bytes generieren - Compiler macht das sehr umstndlich
    __asm("movff	_d_luftmasseL,makeLuftmasse@masse");
    __asm("movff	_d_luftmasseH,makeLuftmasse@masse+1"); 
    masse = masse / 10;        //Berechnung Anzeigewert=Rohwert/100 
                
    DChar[0] = Ch_L;
    DChar[1] = Ch_M;
    DChar[2] = Ch_a;
    DChar[3] = Ch_s;
    DChar[4] = Ch_s;
    DChar[5] = Ch_e;
    DChar[6] = Ch_;
    DChar[12] = Ch_k;
    DChar[13] = Ch_g;
    DChar[14] = Ch_Quer;
    DChar[15] = Ch_h;
    if (GET_BIT(timeout, to_kl_ecua1)) {
        DChar[7] = Ch_nc;
        DChar[8] = Ch_nc;
        DChar[9] = Ch_nc;
        DChar[10] = Ch_nc;
        DChar[11] = Ch_nc;
    } else {
        DChar[7] = getGang(d_gang);
        SET_BIT(flags,f_lze);
        DChar[8] = GetChar(masse,1000);
        DChar[9] = GetChar(ergebnis,100);
        DChar[10] = GetChar(ergebnis,10);
        DChar[11] = GetChar(ergebnis,1);
    }
    return Kons_SendFrame(DChar);      
}

//Erstellt und bermittelt den Screen zur Lambdaanzeige, liefert Tastenzustand zurck
unsigned char makeLambda(void) {
    //"Lambda N"
	//"  0,00  "
    unsigned char DChar[16];
    unsigned char lambda;
    unsigned short tmp;
    lambda = TMR0L;      //Dummy - Das ist leider ntig, sonst optimiert der Compiler erg weg, weil er nicht in den Assembly-Code schaut
    tmp = d_lambda * 200;     //Berechnung: Anzeigewert=Rohwert*200/256
    __asm("movff	makeLambda@tmp+1,makeLambda@lambda");  //lambda = tmp/256    
    DChar[0] = tmp;     //Dummy - Das ist leider ntig, sonst optimiert der Compiler tmp weg, weil er nicht in den Assembly-Code schaut
            
    DChar[0] = Ch_L;
    DChar[1] = Ch_a;
    DChar[2] = Ch_m;
    DChar[3] = Ch_b;
    DChar[4] = Ch_d;
    DChar[5] = Ch_a;
    DChar[6] = Ch_;
    DChar[8] = Ch_;
    DChar[9] = Ch_;
    DChar[11] = Ch_Komma;
    DChar[14] = Ch_;
    DChar[15] = Ch_;
    if (GET_BIT(timeout, to_kl_ecua1)) {
        DChar[7] = Ch_nc;
        DChar[10] = Ch_nc;
        DChar[12] = Ch_nc;
        DChar[13] = Ch_nc;
    } else {
        DChar[7] = getGang(d_gang);
        DChar[10] = GetChar(lambda,100);
        DChar[12] = GetChar(ergebnis,10);
        DChar[13] = GetChar(ergebnis,1);
    }
    return Kons_SendFrame(DChar);      
}

//Stellt die Verbindung zu den ECUs im Motorrad her
unsigned char makeConnect(void) {
    //"Connect "
	//"1:- 2:- "
    unsigned char DChar[16];
    unsigned char ret;
    unsigned char con;
    DChar[0] = Ch_C;
    DChar[1] = Ch_o;
    DChar[2] = Ch_n;
    DChar[3] = Ch_n;
    DChar[4] = Ch_e;
    DChar[5] = Ch_c;
    DChar[6] = Ch_t;
    DChar[7] = Ch_;
    DChar[8] = Ch_1;
    DChar[9] = Ch_DpPkt;
    DChar[10] = Ch_Minus;
    DChar[11] = Ch_;
    DChar[12] = Ch_2;
    DChar[13] = Ch_DpPkt;
    DChar[14] = Ch_Minus;
    DChar[15] = Ch_;
    ret=Kons_SendFrame(DChar);
    
    con=3;
    while (con != 0) {        //So lange in diesem Screen bleiben bis alle ECUs connected sind
        //Wakeup und StartCommunication-Request an MOT-ECU
        KLine_SendWakeup();
        unsigned char w1[] = KL_ECUSTARTCOM;     //Senden des StartCommunication-Request fr ECU - Response sind 7 Byte
        KLine_SendFrame(w1, sizeof(w1)); 
        KLine_RecvFrame(klineresp, 7);
        if (klineresp[6]==0xC5) {           //Geprft wird nur das Checksum-Byte, wenn das dem einer erfolgreichen response entspricht, gilt die Verbindung als erfolgreich
            CLEAR_BIT(con, 0);
            DChar[10] = Ch_Check;
        } else {
            SET_BIT(con, 0);
            DChar[10] = Ch_X;
        }       
        while (!T0_Overflow);   //Warte bis TMR0 berluft - auch als Delay fr nchste Anfrage 
        TMR0_Reset();
        ret=Kons_SendFrame(DChar);          //Display-Update je nach Connection-Ergebnis
    
        //Wakeup und StartCommunication-Request an KOMBI-Instrument
        KLine_SendWakeup();
        unsigned char w2[] = KL_KOMSTARTCOM;     //Senden des StartCommunication-Request fr KOMBI - Response sind 7 Byte
        KLine_SendFrame(w2, sizeof(w2)); 
        KLine_RecvFrame(klineresp, 7);
        if (klineresp[6]==0xE6) {           //Geprft wird nur das Checksum-Byte, wenn das dem einer erfolgreichen response entspricht, gilt die Verbindung als erfolgreich
            CLEAR_BIT(con, 1);
            DChar[14] = Ch_Check;
        } else {
            SET_BIT(con, 1);
            DChar[14] = Ch_X;
        }
        while (!T0_Overflow);   //Warte bis TMR0 berluft - auch als Delay fr nchste Anfrage 
        TMR0_Reset();
        ret=Kons_SendFrame(DChar);          //Display-Update je nach Connection-Ergebnis
    }
    SET_BIT(flags, f_connected);
    return 0;      
}

//Basisabfrage ob Fehlercodes gelesen werden sollen
unsigned char makeDTC1(void) {
    //"N   DTCs"
	//"J lesen?"
    unsigned char DChar[16];
    CLEAR_BIT(flags,f_dtcda);     //DTCs als noch nicht eingeholt markieren - sorgt dafr, dass bei der ersten Anzeige von makeDTC2 zuerst die DTCs neu eingeholt werden
    DChar[0] = Ch_N;
    DChar[1] = Ch_;
    DChar[2] = Ch_;
    DChar[3] = Ch_;
    DChar[4] = Ch_D;
    DChar[5] = Ch_T;
    DChar[6] = Ch_C;
    DChar[7] = Ch_s;
    DChar[8] = Ch_J;
    DChar[9] = Ch_;
    DChar[10] = Ch_l;
    DChar[11] = Ch_e;
    DChar[12] = Ch_s;
    DChar[13] = Ch_e;
    DChar[14] = Ch_n;
    DChar[15] = Ch_Frage;
    return Kons_SendFrame(DChar);     
}

//DTCs von den Gerten einholen und bersicht anzeigen
unsigned char makeDTC2(void) {
    //"Anz.DTCs"
	//">   0000"
    unsigned char DChar[16];
    unsigned short DTCanz;
    unsigned char ret;
    unsigned char wreq[] = KL_ABSSTARTCOM;      //Universeller Buffer fr Wakeup-Requests (kann leider nicht durch individuelle Deklarationen mit den defines erfolgen - braucht zu viel Speicher, erzeugt Banking-Fehler 
    unsigned char dtcreq[] = KL_ABSGETDTC;    //Universeller Buffer fr DTC-Requests (kann leider nicht durch individuelle Deklarationen mit den defines erfolgen - braucht zu viel Speicher, erzeugt Banking-Fehler
    CLEAR_BIT(flags, f_connected);      //Dateneinholung beenden - nach Einholung und Anzeige der DTCs werden sich die Gerte eh von alleine disconnected haben, d.h. ein Neu-Connect wird sowieso ntig sein 
    DChar[0] = Ch_A;
    DChar[1] = Ch_n;
    DChar[2] = Ch_z;
    DChar[3] = Ch_Punkt;
    DChar[4] = Ch_D;
    DChar[5] = Ch_T;
    DChar[6] = Ch_C;
    DChar[7] = Ch_s;
    DChar[8] = Ch_PfeilR;
    DChar[9] = Ch_;
    DChar[10] = Ch_;
    DChar[11] = Ch_;
    if (!GET_BIT(flags,f_dtcda)) {      //Wenn DTCs noch nicht in einem vorherigen Anzeige-Durchgang eingeholt wurden, DTCs erst mal abfragen
        DTCnum=5;                       //Erster Detail-Screen nach spterem Druck auf untere Taste soll der fr ABS sein
        DChar[11] = Ch_Punkt;
        DChar[12] = Ch_Punkt;
        DChar[13] = Ch_Punkt;
        DChar[14] = Ch_Punkt;
        DChar[15] = Ch_Punkt;
        ret=Kons_SendFrame(DChar);
        //Anzahl DTCs von ABS einholen
        KLine_SendWakeup();     //Wakeup und StartCommunication-Request an ABS
        //wreq kann so bleiben - ist schon richtiges Setup fr ABS: {0x81, 0x29, 0xF1, 0x81, 0x1C}   
        KLine_SendFrame(wreq, sizeof(wreq));    //Senden des StartCommunication-Request fr ABS - Response sind 7 Byte
        KLine_RecvFrame(klineresp, 7);
        if (klineresp[6]==0xAE) {           //Geprft wird nur das Checksum-Byte, wenn das dem einer erfolgreichen response entspricht, gilt die Verbindung als erfolgreich
            wait55ms();                       //55ms warten vor dem nchsten request (minimale Zeit im "normal timing mode")
            //dtcreq kann so bleiben - ist schon richtiges Setup fr ABS: {0x84, 0x29, 0xF1, 0x18, 0x02, 0xFF, 0xFF, 0xB6}     
            KLine_SendFrame(dtcreq, sizeof(dtcreq));    //Senden des DTC-Request fr ABS - Response sind mindestens 6 Byte
            TMR0_Reset();                               //Response kann durchaus lnger dauern - um vorzeitigen Timeout zu verhindern
            KLine_RecvFrame(klineresp, 6);
            if (klineresp[3]==0x58) {       //Geprft wird nur das Response-Byte (0x18+0x40=0x58)
                DChar[11] = Ch_;
                dtc_abs = klineresp[4];     //Fehler-Anzahl liegt in Byte 4
            } else {
                DChar[11] = Ch_x;
                dtc_abs = 0;    
            }
        } else {
            DChar[11] = Ch_X;
            dtc_abs = 0; 
        } 
        ret=Kons_SendFrame(DChar);  
        wait200ms();           //Es knnen durchaus mehr als 6 Byte ankommen (je nach Anzahl Fehler). Diese zustzlichen Bytes sollen ignoriert werden, es soll aber auch kein neuer Request derweil rausgehen
        //Anzahl DTCs von DWA einholen
        TMR0_Reset();            //Vorherige DTC-Response kann durchaus lnger gedauert haben - um vorzeitigen Timeout zu verhindern
        KLine_SendWakeup();     //Wakeup und StartCommunication-Request an DWA
        wreq[1]=0x41;       //wreq ndern zu {0x81, 0x41, 0xF1, 0x81, 0x34}
        wreq[4]=0x34;
        KLine_SendFrame(wreq, sizeof(wreq)); //Senden des StartCommunication-Request fr DWA - Response sind 7 Byte
        KLine_RecvFrame(klineresp, 7);
        if (klineresp[6]==0x70) {           //Geprft wird nur das Checksum-Byte, wenn das dem einer erfolgreichen response entspricht, gilt die Verbindung als erfolgreich 
            wait55ms();                       //55ms warten vor dem nchsten request (minimale Zeit im "normal timing mode")
            dtcreq[1]=0x41;       //dtcreq ndern zu {0x84, 0x41, 0xF1, 0x18, 0x02, 0xFF, 0xFF, 0xCE}
            dtcreq[7]=0xCE;
            KLine_SendFrame(dtcreq, sizeof(dtcreq));      //Senden des DTC-Request fr DWA - Response sind mindestens 6 Byte
            TMR0_Reset();                               //Response kann durchaus lnger dauern - um vorzeitigen Timeout zu verhindern
            KLine_RecvFrame(klineresp, 6);
            if (klineresp[3]==0x58) {       //Geprft wird nur das Response-Byte (0x18+0x40=0x58)
                DChar[12] = Ch_;
                dtc_dwa = klineresp[4];     //Fehler-Anzahl liegt in Byte 4
            } else {
                DChar[12] = Ch_x;
                dtc_dwa = 0;    
            }
        } else {
            DChar[12] = Ch_X;
            dtc_dwa = 0; 
        } 
        ret=Kons_SendFrame(DChar); 
        wait200ms();           //Es knnen durchaus mehr als 6 Byte ankommen (je nach Anzahl Fehler). Diese zustzlichen Bytes sollen ignoriert werden, es soll aber auch kein neuer Request derweil rausgehen
        //Anzahl DTCs von KOMBI einholen
        TMR0_Reset();            //Vorherige DTC-Response kann durchaus lnger gedauert haben - um vorzeitigen Timeout zu verhindern
        KLine_SendWakeup();     //Wakeup und StartCommunication-Request an KOMBI
        wreq[1]=0x60;       //wreq ndern zu {0x81, 0x60, 0xF1, 0x81, 0x53}
        wreq[4]=0x53;
        KLine_SendFrame(wreq, sizeof(wreq));        //Senden des StartCommunication-Request fr KOMBI - Response sind 7 Byte
        KLine_RecvFrame(klineresp, 7);
        if (klineresp[6]==0xE6) {           //Geprft wird nur das Checksum-Byte, wenn das dem einer erfolgreichen response entspricht, gilt die Verbindung als erfolgreich
            wait55ms();                       //55ms warten vor dem nchsten request (minimale Zeit im "normal timing mode")
            dtcreq[1]=0x60;       //dtcreq ndern zu {0x84, 0x60, 0xF1, 0x18, 0x02, 0xFF, 0xFF, 0xED}
            dtcreq[7]=0xED;
            KLine_SendFrame(dtcreq, sizeof(dtcreq));      //Senden des DTC-Request fr KOMBI - Response sind mindestens 6 Byte
            TMR0_Reset();                               //Response kann durchaus lnger dauern - um vorzeitigen Timeout zu verhindern
            KLine_RecvFrame(klineresp, 6);
            if (klineresp[3]==0x58) {       //Geprft wird nur das Response-Byte (0x18+0x40=0x58)
                DChar[13] = Ch_;
                dtc_kom = klineresp[4];     //Fehler-Anzahl liegt in Byte 4
            } else {
                DChar[13] = Ch_x;
                dtc_kom = 0;    
            }
        } else {
            DChar[13] = Ch_X;
            dtc_kom = 0; 
        }       
        ret=Kons_SendFrame(DChar); 
        wait200ms();           //Es knnen durchaus mehr als 6 Byte ankommen (je nach Anzahl Fehler). Diese zustzlichen Bytes sollen ignoriert werden, es soll aber auch kein neuer Request derweil rausgehen
        //Anzahl DTCs von MOTECU einholen
        TMR0_Reset();            //Vorherige DTC-Response kann durchaus lnger gedauert haben - um vorzeitigen Timeout zu verhindern
        KLine_SendWakeup();     //Wakeup und StartCommunication-Request an MOT-ECU
        wreq[1]=0x12;       //wreq ndern zu {0x81, 0x12, 0xF1, 0x81, 0x05}
        wreq[4]=0x05;
        KLine_SendFrame(wreq, sizeof(wreq));        //Senden des StartCommunication-Request fr ECU - Response sind 7 Byte 
        KLine_RecvFrame(klineresp, 7);
        if (klineresp[6]==0xC5) {           //Geprft wird nur das Checksum-Byte, wenn das dem einer erfolgreichen response entspricht, gilt die Verbindung als erfolgreich
            wait55ms();                       //55ms warten vor dem nchsten request (minimale Zeit im "normal timing mode")
            dtcreq[1]=0x12;       //dtcreq ndern zu {0x84, 0x12, 0xF1, 0x18, 0x02, 0xFF, 0xFF, 0x9F}
            dtcreq[7]=0x9F;
            KLine_SendFrame(dtcreq, sizeof(dtcreq));      //Senden des DTC-Request fr ECU - Response sind mindestens 6 Byte
            TMR0_Reset();                               //Response kann durchaus lnger dauern - um vorzeitigen Timeout zu verhindern
            KLine_RecvFrame(klineresp, 6);
            if (klineresp[3]==0x58) {       //Geprft wird nur das Response-Byte (0x18+0x40=0x58)
                DChar[14] = Ch_;
                dtc_ecu = klineresp[4];     //Fehler-Anzahl liegt in Byte 4
            } else {
                DChar[14] = Ch_x;
                dtc_ecu = 0;    
            }
        } else {
            DChar[14] = Ch_X;
            dtc_ecu = 0; 
        }       
        ret=Kons_SendFrame(DChar); 
        wait200ms();           //Es knnen durchaus mehr als 6 Byte ankommen (je nach Anzahl Fehler). Diese zustzlichen Bytes sollen ignoriert werden, es soll aber auch kein neuer Request derweil rausgehen
        //Anzahl DTCs von ZFE einholen
        TMR0_Reset();            //Vorherige DTC-Response kann durchaus lnger gedauert haben - um vorzeitigen Timeout zu verhindern
        KLine_SendWakeup();     //Wakeup und StartCommunication-Request an ZFE
        wreq[1]=0x72;       //wreq ndern zu {0x81, 0x72, 0xF1, 0x81, 0x65}
        wreq[4]=0x65;
        KLine_SendFrame(wreq, sizeof(wreq));        //Senden des StartCommunication-Request fr ZFE - Response sind 7 Byte
        KLine_RecvFrame(klineresp, 7);
        if (klineresp[6]==0xA1) {           //Geprft wird nur das Checksum-Byte, wenn das dem einer erfolgreichen response entspricht, gilt die Verbindung als erfolgreich
            wait55ms();                       //55ms warten vor dem nchsten request (minimale Zeit im "normal timing mode")
            dtcreq[1]=0x72;       //dtcreq ndern zu {0x84, 0x72, 0xF1, 0x18, 0x02, 0xFF, 0xFF, 0xFF}
            dtcreq[7]=0xFF;
            KLine_SendFrame(dtcreq, sizeof(dtcreq));      //Senden des DTC-Request fr ZFE - Response sind mindestens 6 Byte
            TMR0_Reset();                               //Response kann durchaus lnger dauern - um vorzeitigen Timeout zu verhindern
            KLine_RecvFrame(klineresp, 6);
            if (klineresp[3]==0x58) {       //Geprft wird nur das Response-Byte (0x18+0x40=0x58)
                DChar[15] = Ch_Punkt;
                dtc_zfe = klineresp[4];     //Fehler-Anzahl liegt in Byte 4
            } else {
                DChar[15] = Ch_x;
                dtc_zfe = 0;    
            }
        } else {
            DChar[15] = Ch_X;
            dtc_zfe = 0; 
        } 
        ret=Kons_SendFrame(DChar); 
        wait200ms();           //Es knnen durchaus mehr als 6 Byte ankommen (je nach Anzahl Fehler). Diese zustzlichen Bytes sollen ignoriert werden, es soll aber auch kein neuer Request derweil rausgehen
        SET_BIT(flags,f_dtcda);     //DTCs sind eingeholt - beim nchsten Anzeigedurchgang DTCs nicht nochmal einholen sondern nur Gesamtanzahl anzeigen
    } 
    //Gesamt-Anzahl anzeigen
    DTCanz = dtc_abs + dtc_dwa + dtc_kom + dtc_ecu + dtc_zfe;
    SET_BIT(flags,f_lze);
    DChar[12] = GetChar(DTCanz,1000);
    DChar[13] = GetChar(ergebnis,100);
    DChar[14] = GetChar(ergebnis,10);
    DChar[15] = GetChar(ergebnis,1);
    return Kons_SendFrame(DChar);  
}

//Fr jede ECU einzeln: Anzahl DTCs anzeigen
unsigned char makeDTC3(void) {
    //"000 DTCs"
	//">  KOMBI"
    unsigned char DChar[16];
    DChar[3] = Ch_;
    DChar[4] = Ch_D;
    DChar[5] = Ch_T;
    DChar[6] = Ch_C;
    DChar[7] = Ch_s;
    DChar[8] = Ch_PfeilR;
    DChar[9] = Ch_;
    DChar[10] = Ch_;
    SET_BIT(flags,f_lze);
    if (DTCnum == 0) DTCnum=5;
    switch(DTCnum) {
        case 1:         //ZFE
            DChar[11] = Ch_;
            DChar[12] = Ch_;
            DChar[13] = Ch_Z;
            DChar[14] = Ch_F;
            DChar[15] = Ch_E;
            DChar[0] = GetChar(dtc_zfe,100);
            break;
        case 2:         //KOMBI
            DChar[11] = Ch_K;
            DChar[12] = Ch_O;
            DChar[13] = Ch_M;
            DChar[14] = Ch_B;
            DChar[15] = Ch_I;
            DChar[0] = GetChar(dtc_kom,100);
            break;
        case 3:         //ECU
            DChar[11] = Ch_;
            DChar[12] = Ch_;
            DChar[13] = Ch_E;
            DChar[14] = Ch_C;
            DChar[15] = Ch_U;
            DChar[0] = GetChar(dtc_ecu,100);
            break;
        case 4:         //DWA
            DChar[11] = Ch_;
            DChar[12] = Ch_;
            DChar[13] = Ch_D;
            DChar[14] = Ch_W;
            DChar[15] = Ch_A;
            DChar[0] = GetChar(dtc_dwa,100);
            break;
        case 5:         //ABS
            DChar[11] = Ch_;
            DChar[12] = Ch_;
            DChar[13] = Ch_A;
            DChar[14] = Ch_B;
            DChar[15] = Ch_S;
            DChar[0] = GetChar(dtc_abs,100);
            break;
        default:        //Nicht existierende screenID
            DChar[11] = Ch_;
            DChar[12] = Ch_F;
            DChar[13] = Ch_A;
            DChar[14] = Ch_I;
            DChar[15] = Ch_L;
            DChar[0] = Ch_0;
            ergebnis=0;
    }  
    DChar[1] = GetChar(ergebnis,10);
    DChar[2] = GetChar(ergebnis,1);
    return Kons_SendFrame(DChar);     
}
