Option Strict Off
Option Explicit On
Imports System.IO
Imports System.IO.Ports
Imports System.Drawing.Drawing2D
Imports MSCommLib

Friend Class MainForm
    Inherits System.Windows.Forms.Form
    Dim WithEvents MSComm1 As New MSComm

    Const KALFILE As String = "Wuchtbock.kal"
    Const KALTEXT As String = "Wuchtbock Kalibrierungsfile"
    Const CONNECT As String = "Connect"
    Const DISCONNECT As String = "Disconnect"
    Const SAVE As String = "Save"
    Const FILE As String = "File"
    Const MESSTEXT As String = "Messung luft. "
    Const START As Byte = &H0           'Startkommando an PIC
    Const STOPP As Byte = &HFF          'Stoppkommando an PIC
    Const STDKALS1 As Double = 241      'Standardkalibrierung Kanal 1: LSB/Gramm
    Const STDKALS2 As Double = 229      'Standardkalibrierung Kanal 2: LSB/Gramm
    Const MMLEER As Byte = 0            'Messmodi
    Const MMKALIB As Byte = 1
    Const MMWUCHT As Byte = 2
    Const RANDL As Byte = 1             'Linker Rand in Anzeigen
    Const RANDR As Byte = 1             'Rechter Rand in Anzeigen
    Const RANDO As Byte = 5             'Oberer Rand in Anzeigen
    Const RANDU As Byte = 5             'Unterer Rand in Anzeigen
    Const ANZSAMPKALIB As Integer = 3200 'Anzahl Samples fr Kalibrierungsmessungen: 3200 sind ca. 2 Sekunden (2000 Byte Position (@ 1kHz) + 200*6=1200 Byte Messwerte (@ ca.100Hz))
    Const ANZSAMPMESS As Integer = 8000 'Anzahl Samples fr Unwuchtmessungen: 8000 sind ca. 5 Sekunden (5000 Byte Position (@ 1kHz) + 500*6=3000 Byte Messwert (@ ca.100Hz)

    Structure Messdatum
        Dim Index As Integer            'Index in Positionsdaten-Array zu dem diese Messdaten zeitlich gehren
        'Dim SensData(1) As Integer       '24-Bit Messwert fr Kanal 0 und 1 - geht leider nicht so einfach mit den ReDims whrend Laufzeit
        Dim Sensor1 As Integer           '24-Bit Messwert Kanal 1
        Dim Sensor2 As Integer           '24-Bit Messwert Kanal 2
    End Structure

    Structure SinusPeriode
        Dim Amplitude As Integer         'Amplitude der Sinusperiode in Mess-Rohwerten
        Dim Dauer As Double             'Dauer der Periode in Messdaten-Indexes
        Dim start As Double             'Start-Index der Periode (Nulldurchgang)
        Dim ende As Double              'End-Index der Periode (Nulldurchgang)
    End Structure

    Dim connected As Boolean
    Dim KalibS1, KalibS2 As Double      'Kalibrierungswerte (LSB/Gramm) der beiden Sensoren
    Dim LeerS1, LeerS2 As Double        'Zwischenvariablen fr die Kalibrierungsmessung
    Dim RadRadius As Double             'Radius des zu vermessenden Rads
    Dim DataPos() As Byte               'Array der Daten vom Positionssensor
    Dim DataMess() As Messdatum         'Array der Daten von den Messsensoren
    Dim zustand As Byte                 'Aktueller Zustand im Ablaufautomaten
    Dim nextpos, nextneg As Byte        'Jeweils nchster Zustand, je nachdem welche Taste gedrckt wird
    Dim messmode As Byte                'Gewnschter Messmodus (Leermessung, Kalibrierungsmessung, Wuchtmessung)

    Private Sub MainForm_Load(ByVal eventSender As System.Object, ByVal eventArgs As System.EventArgs) Handles MyBase.Load
        FillCOMPortList()
        B_Connect.Text = CONNECT
        Timer_TUpdate.Enabled = False
        MacheZustand(0)
        B_Pos.Enabled = False
        B_Neg.Enabled = False
        Clear_Displays()
        KalibS1 = STDKALS1
        KalibS2 = STDKALS2
        LoadKalibration()
        Me.Show()
    End Sub

    Private Sub MainForm_Closed(ByVal eventSender As System.Object, ByVal eventArgs As System.Windows.Forms.FormClosedEventArgs) Handles Me.FormClosed
        If connected = True And C_COMPorts.Text <> FILE Then MSComm1.PortOpen = False
    End Sub

    Private Sub MainForm_Resize(ByVal sender As Object, ByVal e As System.EventArgs) Handles Me.Resize
        'Resize der Mainform: Das meiste wird bereits duch die Anchors gemacht, hier werden nur noch die Heights der Messensor-Displays angepasst
        '23 Ist der Abstand zwischen P_S1 und P_S2, 51 ist der Abstand zw. P_S2 und unterem Rand der Form

        P_S1.Height = (Me.Height - P_S1.Top - 23 - 51) / 2
        P_S2.Height = P_S1.Height
        P_S2.Top = P_S1.Bottom + 23
        Label2.Top = (P_S1.Bottom - P_S1.Top) / 2 + P_S1.Top
        L_S1Gramm.Top = Label2.Bottom + 10
        L_S1Grad.Top = L_S1Gramm.Bottom + 10
        Label5.Top = (P_S2.Bottom - P_S2.Top) / 2 + P_S2.Top
        L_S2Gramm.Top = Label5.Bottom + 10
        L_S2Grad.Top = L_S2Gramm.Bottom + 10
        If zustand = 10 Or zustand = 11 Or zustand = 12 Or zustand = 13 Then DisplayAndProcessData()
    End Sub

    Private Sub FillCOMPortList()
        Dim ports As String() = SerialPort.GetPortNames()
        Dim port As String

        C_COMPorts.Items.Add(FILE)
        For Each port In ports                  'Alle COM-Ports einfuegen
            C_COMPorts.Items.Add(port)
        Next port
        If C_COMPorts.Items.Count > 0 Then
            C_COMPorts.SelectedIndex = 0
            B_Connect.Enabled = True
        End If
    End Sub

    Private Sub B_Connect_Click(ByVal eventSender As System.Object, ByVal eventArgs As System.EventArgs) Handles B_Connect.Click
        If B_Connect.Text = CONNECT Then
            C_COMPorts.Enabled = False
            B_Connect.Text = DISCONNECT
            B_Pos.Enabled = True
            B_Neg.Enabled = True
            If C_COMPorts.Text = FILE Then
                LoadData()
            Else
                With MSComm1
                    .CommPort = Val(Mid(C_COMPorts.SelectedItem, 4))
                    .Settings = "38400,N,8,1"
                    .InBufferSize = Math.Max(ANZSAMPKALIB, ANZSAMPMESS) + 100       'Buffer Size 100 Byte grer als maximal mglich, das sollte reichen um keine Overflow-Fehler zu bekommen
                    .RThreshold = ANZSAMPKALIB          'OnComm-Event auslsen nach sovielen Bytes
                    .InputLen = ANZSAMPKALIB            'Input liest so viele Bytes
                    .PortOpen = True
                End With
                connected = True
            End If
            MacheZustand(0)
        Else
            C_COMPorts.Enabled = True
            B_Connect.Text = CONNECT
            B_Pos.Enabled = False
            B_Neg.Enabled = False
            If C_COMPorts.Text = FILE Then
                'Sonst nichts
            Else
                MSComm1.PortOpen = False
                connected = False
            End If
        End If
    End Sub

    Private Sub P_Pos_MouseClick(ByVal sender As Object, ByVal e As System.Windows.Forms.MouseEventArgs) Handles P_Pos.MouseClick
        'Rechtsclick auf Pos-Diagramm - Abspeichern der aufgenommenen Messdaten
        If zustand = 10 And e.Button = Windows.Forms.MouseButtons.Right Then
            SaveData()
        End If
    End Sub

    Private Sub MSComm1_OnComm() Handles MSComm1.OnComm
        Dim Inputstr As String
        Dim n, pindex, startindex As Integer
        Dim bt As Byte
        Dim syncfehler As Boolean

        Select Case MSComm1.CommEvent
            Case MSCommLib.CommEventConstants.comEventOverrun
                MsgBox("Overrun")
            Case MSCommLib.CommEventConstants.comEventRxOver
                MsgBox("RxOver")
            Case MSCommLib.OnCommConstants.comEvReceive
                MSComm1_SendByte(STOPP)           'Stopp-Kommando an PIC senden
                Timer_TUpdate.Enabled = False
                Inputstr = MSComm1.Input            'Bytes aus Buffer lesen (Anzahl der Bytes je nach Messmodus)
                ReDim DataPos(0)
                If messmode = MMKALIB Then          'Die Kalibrierungsmessung soll...
                    startindex = DataMess.Length
                    ReDim Preserve DataMess(DataMess.Length)       'direkt an die Daten der Leermessung anknpfen
                Else                                'ansonsten...
                    ReDim DataMess(0)               'Messdaten-Array neu aufsetzen
                End If
                syncfehler = False
                n = 1
                pindex = 0
                While n <= Inputstr.Length - 8
                    'Einlesen der Daten - Daten liegen danach in den Arrays an Positionen 0...length-1
                    bt = Asc(Mid(Inputstr, n, 1))
                    If bt < 128 Then                'Bit 7 = 0: bermittlung ist nur ein Positionsbyte
                        DataPos(DataPos.Length - 1) = bt
                        ReDim Preserve DataPos(DataPos.Length)
                        n = n + 1
                        pindex = pindex + 1
                    Else                            'Bit 7 = 1: bermittlung ist ein Positionsbyte gefolgt von Messdaten - insgesamt sollte es 3 Datenpakete zu je 3 Byte geben, jeweils zuesrt Positionsbyte gefolgt von 2 Byte Messdaten
                        ReDim Preserve DataPos(DataPos.Length + 2)
                        DataPos(DataPos.Length - 4) = bt - 128      'Aktuelles Byte ist gleich das Positionsbyte des ersten Datenpakets
                        bt = Asc(Mid(Inputstr, n + 3, 1))       '2. Datenpaket beginnt 3 Byte weiter
                        If bt < 128 Then
                            syncfehler = True          'Auch im 2. Datenpaket sollte Bit7=1 sein
                            DataPos(DataPos.Length - 3) = bt
                        Else
                            DataPos(DataPos.Length - 3) = bt - 128
                        End If
                        bt = Asc(Mid(Inputstr, n + 6, 1))       '3. Datenpaket beginnt 6 Byte weiter
                        If bt < 128 Then
                            syncfehler = True          'Auch im 3. Datenpaket sollte Bit7=1 sein
                            DataPos(DataPos.Length - 2) = bt
                        Else
                            DataPos(DataPos.Length - 2) = bt - 128
                        End If
                        DataMess(DataMess.Length - 1).Index = pindex         'Die ber 3 Datenpakete verschickten Messdaten sind alle zumm Zeitindex des ersten Positionsbytes entstanden 
                        DataMess(DataMess.Length - 1).Sensor1 = 256 * 256 * Asc(Mid(Inputstr, n + 1, 1)) + 256 * Asc(Mid(Inputstr, n + 4, 1)) + Asc(Mid(Inputstr, n + 7, 1))   '24Bit verarbeiten, alle 3 Bytes verwenden
                        DataMess(DataMess.Length - 1).Sensor2 = 256 * 256 * Asc(Mid(Inputstr, n + 2, 1)) + 256 * Asc(Mid(Inputstr, n + 5, 1)) + Asc(Mid(Inputstr, n + 8, 1))
                        'DataMess(DataMess.Length - 1).Sensor1 = 256 * Asc(Mid(Inputstr, n + 1, 1)) + Asc(Mid(Inputstr, n + 4, 1))                                               '16Bit verarbeiten, Low-Byte vernachlssigen
                        'DataMess(DataMess.Length - 1).Sensor2 = 256 * Asc(Mid(Inputstr, n + 2, 1)) + Asc(Mid(Inputstr, n + 5, 1))                                               'Die untersten 8 Bit verschwinden eh im Rauschen
                        ReDim Preserve DataMess(DataMess.Length)
                        n = n + 9
                        pindex = pindex + 3
                    End If
                End While
                ReDim Preserve DataPos(DataPos.Length - 2)
                ReDim Preserve DataMess(DataMess.Length - 2)
                If syncfehler = True Then MsgBox("Es gab Synchronistaionsfehler. Messung evtl. fehlerhaft.")
                Select Case messmode
                    Case MMLEER
                        'Durchschnittswerte ber Messwerte ermitteln
                        Dim m, SumS1, SumS2 As Integer
                        SumS1 = 0
                        SumS2 = 0
                        For m = 0 To DataMess.Length - 1
                            SumS1 = SumS1 + DataMess(m).Sensor1
                            SumS2 = SumS2 + DataMess(m).Sensor2
                        Next
                        LeerS1 = SumS1 / DataMess.Length
                        LeerS2 = SumS2 / DataMess.Length
                        MacheZustand(5)     'Wechsel in Zustand 5 (Messwertaufnahme mit Zusatzgewicht)
                    Case MMKALIB
                        'Durchschnittswerte ber Messwerte ermitteln
                        Dim m, SumS1, SumS2 As Integer
                        SumS1 = 0
                        SumS2 = 0
                        For m = startindex To DataMess.Length - 1
                            SumS1 = SumS1 + DataMess(m).Sensor1
                            SumS2 = SumS2 + DataMess(m).Sensor2
                        Next
                        KalibS1 = SumS1 / (DataMess.Length - startindex)
                        KalibS2 = SumS2 / (DataMess.Length - startindex)
                        'Dialog zur Gewichtseingabe
                        Dim gew As Double
                        Dim intext As String
                        intext = InputBox("Gre des Zusatzgewichts in Gramm eingeben", "Eingabe erforderlich", "10,0")
                        If Not Double.TryParse(intext, gew) Then
                            gew = 10.0
                            MsgBox("Ungltige Eingabe. Es werden 10g angenommen.")
                        End If
                        'Endgltige Ermittlung der Kalibrierungswerte
                        DisplayKalib(startindex)
                        gew = gew / 2               'Das Gesamtgewicht verteilt sich auf 2 Sensoren, da Rad ungefhr mittig aufliegt, wird Verhltnis 50/50 angenommen - Ungenauigkeiten rechnen sich durch Kalibrierung eh raus
                        KalibS1 = (KalibS1 - LeerS1) / gew
                        KalibS2 = (KalibS2 - LeerS2) / gew
                        SaveKalibration()
                        MacheZustand(7)     'Wechsel in Zustand 7 (Ende der Kalibrierung)
                    Case MMWUCHT
                        'Wechsel in Zustand 10 (Anzeige der Kurven, Ermittlung und Anzeige der wichtigen Punkte und der Ergebniswerte)
                        MacheZustand(10)
                    Case Else
                        MsgBox("Ungltiger Mess-Modus")
                End Select
        End Select
    End Sub

    Private Sub MSComm1_RXFlush()
        'Leert den RX-Buffer
        Dim altlen As Integer
        Dim inputstr As String

        altlen = MSComm1.InputLen
        MSComm1.InputLen = 0
        inputstr = MSComm1.Input
        MSComm1.InputLen = altlen
    End Sub

    Private Sub MSComm1_SendBytes(ByVal bytes() As Byte)
        Dim outstr As String
        Dim n As Integer

        outstr = ""
        For n = 0 To bytes.Length - 1
            outstr = outstr + Chr(bytes(n))
        Next
        MSComm1.Output = outstr
    End Sub

    Private Sub MSComm1_SendByte(ByVal bt As Byte)
        Dim outstr As String

        outstr = Chr(bt)
        MSComm1.Output = outstr
    End Sub

    Private Sub LoadData()
        'Ldt ein Messdatenfile (Rohdaten)
        'Daten liegen danach in den Arrays an Positionen 0...length-1
        Dim Myfile As String = ""
        Dim reader As StreamReader
        Dim line As String

        With OpenFileDialog
            .Filter = "Wuchtbock-Files (*.kwd)|*.kwd"
            .FilterIndex = 1
            '.InitialDirectory = OpStrings(1)
            .Title = "Wuchtbock-Daten laden"
            .FileName = ""
            If .ShowDialog() = DialogResult.OK Then Myfile = .FileName
        End With
        If Len(Myfile) > 0 Then
            Try
                reader = New StreamReader(Myfile)
            Catch
                MsgBox("Ladefehler", vbOK + vbCritical, "Fehler")
                Exit Sub
            End Try

            Try
                reader.BaseStream.Seek(0, SeekOrigin.Begin)     ' Pointer auf file-Anfang
                line = reader.ReadLine()                               'Erkennungszeile
                If Mid(line, 1, 7) = "1.0:KWD" Then
                    ReDim DataPos(0)
                    ReDim DataMess(0)
                    KalibS1 = CDbl(reader.ReadLine)
                    KalibS2 = CDbl(reader.ReadLine)
                    line = reader.ReadLine()            '"Positionssensor"
                    line = reader.ReadLine()
                    While line <> "Messsensoren"
                        DataPos(DataPos.Length - 1) = CByte(line)
                        ReDim Preserve DataPos(DataPos.Length)
                        line = reader.ReadLine()
                    End While
                    ReDim Preserve DataPos(DataPos.Length - 2)
                    line = reader.ReadLine()
                    Dim items() As String
                    While Not reader.EndOfStream        'Letzte Zeile ("End") ignorieren
                        items = line.Split("|")
                        DataMess(DataMess.Length - 1).Index = CInt(items(0))
                        DataMess(DataMess.Length - 1).Sensor1 = CInt(items(1))
                        DataMess(DataMess.Length - 1).Sensor2 = CInt(items(2))
                        ReDim Preserve DataMess(DataMess.Length)
                        line = reader.ReadLine()
                    End While
                    ReDim Preserve DataMess(DataMess.Length - 2)
                Else
                    MsgBox("Unbekanntes Fileformat")
                End If
                reader.Close()
            Catch
                MsgBox("Lesefehler", vbOK + vbCritical, "Fehler")
                reader.Close()
                Exit Sub
            End Try
        End If
    End Sub

    Private Sub SaveData()
        'Sichert Messdaten (Rohdaten) in ein File
        Dim Myfile As String = ""

        With SaveFileDialog
            .Filter = "Wuchtbock-Files (*.kwd)|*.kwd"
            .FilterIndex = 1
            '.InitialDirectory = OpStrings(1)
            .Title = "Daten in File speichern"
            .FileName = ""
        End With
        If SaveFileDialog.ShowDialog() = DialogResult.OK Then
            Myfile = SaveFileDialog.FileName
            Dim quelle As New FileStream(Myfile, FileMode.Create)
            Dim writer As New StreamWriter(quelle)
            Dim n As Integer
            writer.BaseStream.Seek(0, SeekOrigin.Begin)     ' Pointer auf file-Anfang
            writer.WriteLine("1.0:KWD - Datenfile einer Wuchtbock-Messung")
            writer.WriteLine(KalibS1.ToString)
            writer.WriteLine(KalibS2.ToString)
            writer.WriteLine("Positionssensor")
            For n = 0 To DataPos.Length - 1
                writer.WriteLine(DataPos(n).ToString)
            Next
            writer.WriteLine("Messsensoren")
            For n = 0 To DataMess.Length - 1
                writer.WriteLine(DataMess(n).Index.ToString + "|" + DataMess(n).Sensor1.ToString + "|" + DataMess(n).Sensor2.ToString)
            Next
            writer.WriteLine("End")
            writer.Close()
            quelle.Close()
        End If
    End Sub

    Private Sub LoadKalibration()
        'Ldt die zuletzt gespeicherte Kalibrierung
        Dim reader As StreamReader
        Dim readstring As String

        Try
            reader = New StreamReader(KALFILE)
        Catch
            MsgBox("Ladefehler", vbOK + vbCritical, "Fehler")
            Exit Sub
        End Try

        Try
            reader.BaseStream.Seek(0, SeekOrigin.Begin)     'Pointer auf file-Anfang
            readstring = reader.ReadLine()                  'Erkennungszeile
            If readstring = KALTEXT Then
                KalibS1 = CDbl(reader.ReadLine())
                KalibS2 = CDbl(reader.ReadLine())
            Else
                MsgBox("Unbekanntes Fileformat")
            End If
            reader.Close()
        Catch
            MsgBox("Lesefehler", vbOK + vbCritical, "Fehler")
            reader.Close()
            Exit Sub
        End Try
    End Sub

    Private Sub SaveKalibration()
        'Speichert die aktuelle Kalibrierung
        Dim quelle As New FileStream(KALFILE, FileMode.Create)
        Dim writer As New StreamWriter(quelle)
        writer.BaseStream.Seek(0, SeekOrigin.Begin)     ' Pointer auf file-Anfang
        writer.WriteLine(KALTEXT)
        writer.WriteLine(KalibS1.ToString)
        writer.WriteLine(KalibS2.ToString)
        writer.Close()
        quelle.Close()
    End Sub

    Private Sub MacheZustand(ByVal state As Byte)
        'Die Ablaufanweisungen der Zustandsmaschine, Zustandsnderungen siehe die Routinen fr die zwei Buttons 
        zustand = state
        Select Case state
            Case 0
                T_Anweisung.Text = "Zur Vorbereitung 6 Punkte (scrollbar):" + vbNewLine + "1: Felgendurchmesser ermitteln (von einer Felgenflche, auf der spter die Auswuchtgewichte aufgeklebt werden zur gegenberliegenden)" + vbNewLine + "2: Magnete gegenber an Rad anbringen" + vbNewLine + "3: Achse fest im Rad montieren, Rad mittig auf Wuchtbock auflegen" + vbNewLine + "4: Positionssensor so ausrichten, dass er ca. 5mm Abstand zu der Magnetbahn hat, Achse mit Klemmbrcken fixieren" + vbNewLine + "5: Sicherungsstifte unter den Messensoren entfernen" + vbNewLine + "6: Connecten"
                Clear_Displays()
                B_Pos.Text = "OK"
                nextpos = 1
                B_Neg.Text = "OK"
                nextneg = 1
            Case 1
                T_Anweisung.Text = "Der Reifendurchmesser ist ntig, um das Auswuchtgewicht entsprechend danach auszurichten."
                'Dialog zur Eingabe des Rad-Durchmessers
                Dim intext As String
                intext = InputBox("Durchmesser des Rads in cm eingeben", "Eingabe erforderlich", "42,0")
                If Not Double.TryParse(intext, RadRadius) Then
                    RadRadius = 42.0
                    MsgBox("Ungltige Eingabe. Es werden 42cm angenommen.")
                End If
                RadRadius = RadRadius / 2 / 100     'Umrechnung von Durchmesser in Radius und von cm in Meter
                If connected = True Then        'Bei Connection mit Wuchtbock
                    MacheZustand(2)                      'Kalibrierung vorschlagen
                Else                            'Bei Connection mit File
                    MacheZustand(10)                      'Kalibrierung macht keinen Sinn, gleich weiter zur Anzeige
                End If
            Case 2
                T_Anweisung.Text = "Es sollte immer erst eine Kalibrierung gemacht werden." + vbNewLine + "Evtl ist Radposition nicht ganz mittig, Vernderungen der Sensoren ber Zeit, Temperatur und Gesamtgewicht des Rads, etc."
                B_Pos.Text = "Starten"
                nextpos = 3
                B_Neg.Text = "berspringen"
                nextneg = 8
            Case 3
                T_Anweisung.Text = "Rad muss ruhig aufliegen, dann OK drcken"
                B_Pos.Text = "OK"
                nextpos = 4
                B_Neg.Text = "berspringen"
                nextneg = 8
                If Timer_TUpdate.Enabled = True Then MSComm1_SendByte(STOPP) 'Wenn Messung aktiv ist (durch "Abbrechen" hierher gekommen: Stopp-Kommando an PIC senden
                Timer_TUpdate.Enabled = False
            Case 4
                T_Anweisung.Text = MESSTEXT + "0%"
                B_Pos.Text = "Abbrechen"
                nextpos = 3
                B_Neg.Text = "Abbrechen"
                nextneg = 3
                MSComm1_RXFlush()            'Input-Buffer leeren
                'Kurze Messung zur Aufnahme der Messdaten ohne Zusatzgewicht - Ende und Auswertung automatisch nach Messwertaufnahme
                MSComm1.RThreshold = ANZSAMPKALIB       'OnComm-Event auslsen nach sovielen Bytes
                MSComm1.InputLen = ANZSAMPKALIB         'Input liest so viele Bytes
                messmode = MMLEER
                MSComm1_SendByte(START)      'Start-Kommando an PIC senden
                Timer_TUpdate.Enabled = True    'Startet Messfortschrittsanzeige
                'Weiterschaltung in Zustand 5 (Messwertaufnahme mit Zusatzgewicht) automatisch nach Messwertaufnahme
            Case 5
                T_Anweisung.Text = "Zusatzgewicht (ca. 10g) auf Rad auflegen, dann OK drcken. Im Anschluss genaue Hhe des Zusatzgewichts eingeben."
                B_Pos.Text = "OK"
                nextpos = 6
                B_Neg.Text = "berspringen"
                nextneg = 8
                If Timer_TUpdate.Enabled = True Then MSComm1_SendByte(STOPP) 'Wenn Messung aktiv ist (durch "Abbrechen" hierher gekommen: Stopp-Kommando an PIC senden
                Timer_TUpdate.Enabled = False
            Case 6
                T_Anweisung.Text = MESSTEXT + "0%"
                B_Pos.Text = "Abbrechen"
                nextpos = 5
                B_Neg.Text = "Abbrechen"
                nextneg = 5
                MSComm1_RXFlush()           'Input-Buffer leeren
                'Kurze Messung zur Aufnahme der Messdaten mit Zusatzgewicht - Ende und Auswertung automatisch nach Messwertaufnahme
                MSComm1.RThreshold = ANZSAMPKALIB       'OnComm-Event auslsen nach sovielen Bytes
                MSComm1.InputLen = ANZSAMPKALIB         'Input liest so viele Bytes
                messmode = MMKALIB
                MSComm1_SendByte(START)      'Start-Kommando an PIC senden
                Timer_TUpdate.Enabled = True    'Startet Messfortschrittsanzeige
                'Dialog zur Gewichtseingabe und endgltiger Ermittlung der Kalibrierungswerte erfolgt in MSComm_OnComm, dort auch automatisch Weiterschaltung in Zustand 7
            Case 7
                T_Anweisung.Text = "Kalibrierung ist beendet." + vbNewLine + "Ermittelte Werte sind " + KalibS1.ToString("F1") + " und " + KalibS2.ToString("F1") + "." + vbNewLine + "Bei zentral aufgesetztem Rad sollten die Werte im Rahmen 220-250 liegen." + vbNewLine + "Weitermachen mit Messung?"
                B_Pos.Text = "OK"
                nextpos = 8
                B_Neg.Text = "Kal. wiederholen"
                nextneg = 3
            Case 8
                T_Anweisung.Text = "Aufgelegtes Rad auf ca. 1-2 Umdrehungen/Sekunde bringen, dann OK drcken."
                Clear_Displays()
                B_Pos.Text = "OK"
                nextpos = 9
                B_Neg.Text = "Abbrechen"
                nextneg = 0
                If Timer_TUpdate.Enabled = True Then MSComm1_SendByte(STOPP) 'Wenn Messung aktiv ist (durch "Abbrechen" hierher gekommen: Stopp-Kommando an PIC senden
                Timer_TUpdate.Enabled = False
            Case 9
                T_Anweisung.Text = MESSTEXT + "0%"
                B_Pos.Text = "Abbrechen"
                nextpos = 8
                B_Neg.Text = "Abbrechen"
                nextneg = 8
                MSComm1_RXFlush()            'Input-Buffer leeren
                'Start der Messung - Ende automatisch nachdem genug Messwerte gesammelt wurden 
                MSComm1.RThreshold = ANZSAMPMESS       'OnComm-Event auslsen nach sovielen Bytes
                MSComm1.InputLen = ANZSAMPMESS         'Input liest so viele Bytes
                messmode = MMWUCHT
                MSComm1_SendByte(START)      'Start-Kommando an PIC senden
                Timer_TUpdate.Enabled = True    'Startet Messfortschrittsanzeige
                'Weiterschaltung in Zustand 10 (Anzeige der Kurven, Ermittlung und Anzeige der wichtigen Punkte und der Ergebniswerte) automatisch nach Messwertaufnahme
            Case 10
                T_Anweisung.Text = "Messung ist beendet, siehe Anzeigen." + vbNewLine + "Sehen Kurven vernnftig aus?" + vbNewLine + "Abspeichern der Messung mit Rechtsklick auf Positionsdiagramm"
                DisplayAndProcessData()
                B_Pos.Text = "Sieht gut aus"
                nextpos = 11
                B_Neg.Text = "Wiederholen"
                If connected = True Then        'Bei Connection mit Wuchtbock
                    nextneg = 8                      'Neumessung
                Else                            'Bei Connection mit File
                    nextneg = 0                      'zurck auf Start
                End If
            Case 11
                T_Anweisung.Text = "Zum Anbringen der Auswuchtgewichte:" + vbNewLine + "Rad so drehen, dass der grne Magnet genau am Positionssensor liegt."
                B_Pos.Text = "OK"
                nextpos = 12
                B_Neg.Text = "Abbrechen"
                If connected = True Then        'Bei Connection mit Wuchtbock
                    nextneg = 8                      'Neumessung
                Else                            'Bei Connection mit File
                    nextneg = 10                      'zurck zur Anzeige
                End If
            Case 12
                Dim alphaneg As Double
                alphaneg = (360 - CDbl(L_Grad.Text))
                T_Anweisung.Text = "0-Uhr Position am Rad markieren und diese um angezeigte " + L_Grad.Text + " in Drehrichtung bzw. um " + alphaneg.ToString("F0") + " gegen Drehrichtung verdrehen."
                B_Pos.Text = "OK"
                nextpos = 13
                B_Neg.Text = "Abbrechen"
                If connected = True Then        'Bei Connection mit Wuchtbock
                    nextneg = 8                      'Neumessung
                Else                            'Bei Connection mit File
                    nextneg = 10                      'zurck zur Anzeige
                End If
            Case 13
                T_Anweisung.Text = "Position fr Auswuchtgewichte liegt nun genau auf 6 Uhr, also ganz unten." + vbNewLine + "Klebegewichte (" + L_Gramm.Text + "g) dort anbringen, so mittig wie mglich, am besten gleichmig links und rechts."
                B_Pos.Text = "Kontrollmessung starten"
                If connected = True Then        'Bei Connection mit Wuchtbock
                    nextpos = 8                      'Neumessung
                Else                            'Bei Connection mit File
                    nextpos = 10                      'zurck zur Anzeige
                End If
                B_Neg.Text = "Neues Rad vermessen"
                nextneg = 0
            Case Else
                T_Anweisung.Text = "Unerlaubter Zustand"
                B_Pos.Text = "Reset"
                nextpos = 0
                B_Neg.Text = "Reset"
                nextneg = 0
        End Select
    End Sub

    Private Sub Timer_TUpdate_Tick(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles Timer_TUpdate.Tick
        'Lst alle 0,5 sek aus: Updatet Messfortschritt
        Dim perc As Double

        perc = MSComm1.InBufferCount / MSComm1.RThreshold * 100
        T_Anweisung.Text = MESSTEXT + perc.ToString("F0") + "%"
    End Sub

    Private Sub B_Pos_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles B_Pos.Click
        'Die Zustandsnderungen der Zustandsmaschine, zu den Anweisungen siehe die MacheZustand-Sub

        MacheZustand(nextpos)
    End Sub

    Private Sub B_Neg_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles B_Neg.Click
        'Die Zustandsnderungen der Zustandsmaschine, zu den Anweisungen siehe die MacheZustand-Sub

        MacheZustand(nextneg)
    End Sub

    Private Sub Clear_Displays()
        'Lscht die Displays und die Ergebnisanzeige
        P_Pos.BackColor = Color.Green
        P_Pos.Image = Nothing
        P_S1.BackColor = Color.Green
        P_S1.Image = Nothing
        P_S2.BackColor = Color.Green
        P_S2.Image = Nothing
        L_Gramm.Text = "T.b.d."
        L_Gramm.ForeColor = Color.Red
        L_Grad.Text = "T.b.d."
        L_Grad.ForeColor = Color.Red
        L_S1Gramm.Text = ""
        L_S1Grad.Text = ""
        L_S2Gramm.Text = ""
        L_S2Grad.Text = ""
    End Sub

    Private Sub DisplayKalib(ByVal kindex As Integer)
        'Zeigt die Kalibrierungsmessungen an, kindex ist der Index ab dem die Daten mit Zusatzgewicht im Array liegen
        Dim n As Integer
        Dim penline As Pen
        Dim oldx, oldy, newx, newy As Integer

        penline = New Pen(Color.GreenYellow, 1)

        '***** Anzeige der Sensor1-Daten ********
        Dim S1Max, S1Min As Integer
        Dim S1YPixProVal As Double      'Skalierung in Pixel/Value
        Dim S1YRange, S1Ynull As Double
        Dim bi2 As New Bitmap(P_S1.Width, P_S1.Height)
        'Ermittlung von Min/Max-Werten und Skalierungen
        S1Min = DataMess(0).Sensor1
        S1Max = DataMess(0).Sensor1
        For n = 1 To DataMess.Length - 1
            If DataMess(n).Sensor1 < S1Min Then S1Min = DataMess(n).Sensor1
            If DataMess(n).Sensor1 > S1Max Then S1Max = DataMess(n).Sensor1
        Next
        If S1Max = S1Min Then
            S1YRange = 1
        Else
            S1YRange = S1Max - S1Min
        End If
        S1YPixProVal = (P_S1.Height - RANDO - RANDU) / S1YRange          'Y-Skalierung Pixel/Value
        S1Ynull = S1YPixProVal * S1Max + RANDO                           'Ymax liegt an oberem Anzeigerand, 0-Linie entsprechend unterhalb
        'Anzeige des Graphen
        Using gr As Graphics = Graphics.FromImage(bi2)
            gr.Clear(Color.Green)
            gr.SmoothingMode = SmoothingMode.AntiAlias
            penline.Color = Color.GreenYellow
            oldx = RANDL
            oldy = CInt(S1Ynull - DataMess(0).Sensor1 * S1YPixProVal)
            For n = 1 To DataMess.Length - 1
                newx = CInt(Math.Round(n / DataMess.Length * (P_S1.Width - RANDL - RANDR), 0)) + RANDL
                newy = CInt(S1Ynull - DataMess(n).Sensor1 * S1YPixProVal)
                gr.DrawLine(penline, oldx, oldy, newx, newy)
                oldx = newx
                oldy = newy
            Next
            'Einzeichnen der errechneten Durchschnittswerte
            penline.Color = Color.Red
            newx = CInt(Math.Round((kindex - 1) / DataMess.Length * (P_S1.Width - RANDL - RANDR), 0)) + RANDL
            newy = CInt(S1Ynull - LeerS1 * S1YPixProVal)
            gr.DrawLine(penline, 0, newy, newx, newy)
            newx = CInt(Math.Round(kindex / DataMess.Length * (P_S1.Width - RANDL - RANDR), 0)) + RANDL
            newy = CInt(S1Ynull - KalibS1 * S1YPixProVal)
            gr.DrawLine(penline, newx, newy, P_S1.Width, newy)
        End Using
        P_S1.Image = bi2


        '***** Anzeige der Sensor2-Daten ********
        Dim S2Max, S2Min As Integer
        Dim S2YPixProVal As Double      'Skalierung in Pixel/Value
        Dim S2YRange, S2Ynull As Double
        Dim bi3 As New Bitmap(P_S2.Width, P_S2.Height)
        'Ermittlung von Min/Max-Werten und Skalierungen
        S2Min = DataMess(0).Sensor2
        S2Max = DataMess(0).Sensor2
        For n = 1 To DataMess.Length - 1
            If DataMess(n).Sensor2 < S2Min Then S2Min = DataMess(n).Sensor2
            If DataMess(n).Sensor2 > S2Max Then S2Max = DataMess(n).Sensor2
        Next
        If S2Max = S2Min Then
            S2YRange = 1
        Else
            S2YRange = S2Max - S2Min
        End If
        S2YPixProVal = (P_S2.Height - RANDO - RANDU) / S2YRange          'Y-Skalierung Pixel/Value
        S2Ynull = S2YPixProVal * S2Max + RANDO                           'Ymax liegt an oberem Anzeigerand, 0-Linie entsprechend unterhalb
        'Anzeige des Graphen
        Using gr As Graphics = Graphics.FromImage(bi3)
            gr.Clear(Color.Green)
            gr.SmoothingMode = SmoothingMode.AntiAlias
            penline.Color = Color.GreenYellow
            oldx = RANDL
            oldy = CInt(S2Ynull - DataMess(0).Sensor2 * S2YPixProVal)
            For n = 1 To DataMess.Length - 1
                newx = CInt(Math.Round(n / DataMess.Length * (P_S2.Width - RANDL - RANDR), 0)) + RANDL
                newy = CInt(S2Ynull - DataMess(n).Sensor2 * S2YPixProVal)
                gr.DrawLine(penline, oldx, oldy, newx, newy)
                oldx = newx
                oldy = newy
            Next
            'Einzeichnen der errechneten Durchschnittswerte
            penline.Color = Color.Red
            newx = CInt(Math.Round((kindex - 1) / DataMess.Length * (P_S2.Width - RANDL - RANDR), 0)) + RANDL
            newy = CInt(S2Ynull - LeerS2 * S2YPixProVal)
            gr.DrawLine(penline, 0, newy, newx, newy)
            newx = CInt(Math.Round(kindex / DataMess.Length * (P_S2.Width - RANDL - RANDR), 0)) + RANDL
            newy = CInt(S2Ynull - KalibS2 * S2YPixProVal)
            gr.DrawLine(penline, newx, newy, P_S2.Width, newy)
        End Using
        P_S2.Image = bi3
    End Sub

    Private Sub DisplayAndProcessData()
        'Zeichnet die Graphen der 3 Messreihen (Position, Sens1, Sens2) inkl. wichtiger Punkte und ermittelt die Ergebniswert in Grad und Gramm
        Dim n, m As Integer
        Dim penline As Pen
        Dim oldx, oldy, newx, newy As Integer
        Dim filteredS(1)() As Integer
        Dim glaettung As Byte

        glaettung = 15      'Gewnschte Glttungsbreite fr SesnorDaten

        '***** Anzeige der Positionssensor-Daten ********
        Dim PosMax, PosMin As Byte
        Dim PosYPixProVal As Double      'Skalierung in Pixel/Value
        Dim PosYRange, PosYnull As Double
        Dim bi As New Bitmap(P_Pos.Width, P_Pos.Height)
        'Ermittlung von Min/Max-Werten und Skalierungen
        PosMin = DataPos(0)
        PosMax = DataPos(0)
        For n = 1 To DataPos.Length - 1
            If DataPos(n) < PosMin Then PosMin = DataPos(n)
            If DataPos(n) > PosMax Then PosMax = DataPos(n)
        Next
        If PosMax = PosMin Then
            PosYRange = 1
        Else
            PosYRange = PosMax - PosMin
        End If
        PosYPixProVal = (P_Pos.Height - RANDO - RANDU) / PosYRange          'Y-Skalierung Pixel/Value
        PosYnull = PosYPixProVal * PosMax + RANDO                           'Ymax liegt an oberem Anzeigerand, 0-Linie entsprechend unterhalb
        'Ermittlung der Peak-Werte
        Dim PosPeaks() As Integer
        PosPeaks = GetPosPeakIndexes(PosMin, PosMax)
        'Anzeige des Graphen
        Using gr As Graphics = Graphics.FromImage(bi)
            gr.Clear(Color.Green)
            gr.SmoothingMode = SmoothingMode.AntiAlias
            penline = New Pen(Color.GreenYellow, 1)
            oldx = RANDL
            oldy = CInt(PosYnull - DataPos(0) * PosYPixProVal)
            For n = 1 To DataPos.Length - 1
                newx = CInt(Math.Round(n / DataPos.Length * (P_Pos.Width - RANDL - RANDR), 0)) + RANDL
                newy = CInt(PosYnull - DataPos(n) * PosYPixProVal)
                gr.DrawLine(penline, oldx, oldy, newx, newy)
                oldx = newx
                oldy = newy
            Next
            'Einzeichnen der erkannten Peaks
            penline.Color = Color.Red
            For n = 0 To PosPeaks.Length - 1
                newx = CInt(Math.Round(PosPeaks(n) / DataPos.Length * (P_Pos.Width - RANDL - RANDR), 0)) + RANDL
                gr.DrawLine(penline, newx, 0, newx, P_Pos.Height - 1)
            Next
        End Using
        P_Pos.Image = bi


        '***** Anzeige der Sensor1-Daten ********
        Dim S1Max, S1Min As Integer
        Dim S1YPixProVal As Double      'Skalierung in Pixel/Value
        Dim S1YRange, S1Ynull As Double
        Dim bi2 As New Bitmap(P_S1.Width, P_S1.Height)
        'Ermittlung von Min/Max-Werten und Skalierungen
        S1Min = DataMess(0).Sensor1
        S1Max = DataMess(0).Sensor1
        For n = 1 To DataMess.Length - 1
            If DataMess(n).Sensor1 < S1Min Then S1Min = DataMess(n).Sensor1
            If DataMess(n).Sensor1 > S1Max Then S1Max = DataMess(n).Sensor1
        Next
        If S1Max = S1Min Then
            S1YRange = 1
        Else
            S1YRange = S1Max - S1Min
        End If
        S1YPixProVal = (P_S1.Height - RANDO - RANDU) / S1YRange          'Y-Skalierung Pixel/Value
        S1Ynull = S1YPixProVal * S1Max + RANDO                           'Ymax liegt an oberem Anzeigerand, 0-Linie entsprechend unterhalb
        'Ermittlung der geglaetteten Kurve 
        ReDim filteredS(0)(DataMess.Length - 1)
        filteredS(0) = GetFilteredCurve(1, glaettung)
        'Ermittlung der Nulllinie
        Dim S1NullStart, S1NullEnde As Integer
        Dim S1NullWert As Double
        S1NullStart = GetFirstHighLowInFilteredS(filteredS(0), glaettung, True)     'Ersten Hochpunkt suchen
        S1NullEnde = GetFirstHighLowInFilteredS(filteredS(0), glaettung, False)     'Ersten Tiefpunkt suchen
        If S1NullStart < S1NullEnde Then        'Hochpunkt kommt zuerst: Nulllinie von erstem Hochpunkt zu letzten Hochpunkt
            S1NullEnde = GetLastHighLowInFilteredS(filteredS(0), glaettung, True)
        Else                                    'Tiefpunkt kommt zuerst: Nulllinie von erstem Tiefpunkt zu letzten Tiefpunkt
            S1NullStart = S1NullEnde
            S1NullEnde = GetLastHighLowInFilteredS(filteredS(0), glaettung, False)
        End If
        S1NullWert = GetNullwertFromFilteredS(filteredS(0), S1NullStart, S1NullEnde)
        'Ermittlung der Nulldurchgaenge
        Dim S1NullIndexes() As Double
        S1NullIndexes = GetNullDurchgaengeInFilteredS(filteredS(0), S1NullStart, S1NullEnde, S1NullWert)    'Indexe sind auf Basis der Messdaten, aber evtl Fliekomma
        'Ermittlung der Parameter der einzelnen Sinusperioden
        Dim S1Periods() As SinusPeriode
        S1Periods = GetSinusPeriods(S1NullIndexes, S1NullWert, 1)
        'Anzeige des Graphen
        Using gr As Graphics = Graphics.FromImage(bi2)
            gr.Clear(Color.Green)
            gr.SmoothingMode = SmoothingMode.AntiAlias
            penline.Color = Color.GreenYellow
            oldx = RANDL
            oldy = CInt(S1Ynull - DataMess(0).Sensor1 * S1YPixProVal)
            For n = 1 To DataMess.Length - 1
                newx = CInt(Math.Round(DataMess(n).Index / DataPos.Length * (P_S1.Width - RANDL - RANDR), 0)) + RANDL   'X-Skalierung orientiert sich an den Indexen in der Pos-Grafik
                newy = CInt(S1Ynull - DataMess(n).Sensor1 * S1YPixProVal)
                gr.DrawLine(penline, oldx, oldy, newx, newy)
                oldx = newx
                oldy = newy
            Next
            'Einzeichnen der gegltteten Kurve, inkl. Nulllinie und Grenzen der Nulllinienfindung
            penline.Color = Color.Blue
            oldx = RANDL
            oldy = CInt(S1Ynull - filteredS(0)(0) * S1YPixProVal)
            For n = 1 To DataMess.Length - 1
                newx = CInt(Math.Round(DataMess(n).Index / DataPos.Length * (P_S1.Width - RANDL - RANDR), 0)) + RANDL   'X-Skalierung orientiert sich an den Indexen in der Pos-Grafik
                newy = CInt(S1Ynull - filteredS(0)(n) * S1YPixProVal)
                gr.DrawLine(penline, oldx, oldy, newx, newy)
                oldx = newx
                oldy = newy
            Next
            oldx = CInt(Math.Round(DataMess(S1NullStart).Index / DataPos.Length * (P_S1.Width - RANDL - RANDR), 0)) + RANDL   'X-Skalierung orientiert sich an den Indexen in der Pos-Grafik
            gr.DrawLine(penline, oldx, 0, oldx, P_S1.Height - 1)
            newx = CInt(Math.Round(DataMess(S1NullEnde).Index / DataPos.Length * (P_S1.Width - RANDL - RANDR), 0)) + RANDL   'X-Skalierung orientiert sich an den Indexen in der Pos-Grafik
            gr.DrawLine(penline, newx, 0, newx, P_S1.Height - 1)
            newy = CInt(S1Ynull - S1NullWert * S1YPixProVal)
            gr.DrawLine(penline, oldx, newy, newx, newy)
            'Einzeichnen der Nulldurchgnge und der finalen Sinuse
            penline.Color = Color.Red
            For n = 0 To S1NullIndexes.Length - 1
                newx = CInt(Math.Round(GetPosIndexFromMessIndex(S1NullIndexes(n)) / DataPos.Length * (P_S1.Width - RANDL - RANDR), 0)) + RANDL   'X-Skalierung orientiert sich an den Indexen in der Pos-Grafik, NullIndexes sind auf Basis der Messdaten
                gr.DrawLine(penline, newx, 0, newx, P_S1.Height - 1)
            Next
            For n = 0 To S1Periods.Length - 1
                oldx = CInt(Math.Round(GetPosIndexFromMessIndex(S1Periods(n).start) / DataPos.Length * (P_S1.Width - RANDL - RANDR), 0)) + RANDL   'X-Skalierung orientiert sich an den Indexen in der Pos-Grafik, StartIndex ist auf Basis der Messdaten
                oldy = CInt(S1Ynull - S1NullWert * S1YPixProVal)
                For m = GetPosIndexFromMessIndex(S1Periods(n).start + 1) To GetPosIndexFromMessIndex(S1Periods(n).ende)
                    newx = CInt(Math.Round(m / DataPos.Length * (P_S1.Width - RANDL - RANDR), 0)) + RANDL                       'X-Skalierung orientiert sich an den Indexen in der Pos-Grafik
                    newy = CInt(S1Ynull - SinusWert(S1Periods(n).Amplitude, m, GetPosIndexFromMessIndex(S1Periods(n).start), GetPosIndexFromMessIndex(S1Periods(n).ende), S1NullWert) * S1YPixProVal)
                    gr.DrawLine(penline, oldx, oldy, newx, newy)
                    oldx = newx
                    oldy = newy
                Next
            Next
        End Using
        P_S1.Image = bi2


        '***** Anzeige der Sensor2-Daten ********
        Dim S2Max, S2Min As Integer
        Dim S2YPixProVal As Double      'Skalierung in Pixel/Value
        Dim S2YRange, S2Ynull As Double
        Dim bi3 As New Bitmap(P_S2.Width, P_S2.Height)
        'Ermittlung von Min/Max-Werten und Skalierungen
        S2Min = DataMess(0).Sensor2
        S2Max = DataMess(0).Sensor2
        For n = 1 To DataMess.Length - 1
            If DataMess(n).Sensor2 < S2Min Then S2Min = DataMess(n).Sensor2
            If DataMess(n).Sensor2 > S2Max Then S2Max = DataMess(n).Sensor2
        Next
        If S2Max = S2Min Then
            S2YRange = 1
        Else
            S2YRange = S2Max - S2Min
        End If
        S2YPixProVal = (P_S2.Height - RANDO - RANDU) / S2YRange          'Y-Skalierung Pixel/Value
        S2Ynull = S2YPixProVal * S2Max + RANDO                           'Ymax liegt an oberem Anzeigerand, 0-Linie entsprechend unterhalb
        'Ermittlung der geglaetteten Kurve 
        ReDim filteredS(1)(DataMess.Length - 1)
        filteredS(1) = GetFilteredCurve(2, glaettung)
        'Ermittlung der Nulllinie
        Dim S2NullStart, S2NullEnde As Integer
        Dim S2NullWert As Double
        S2NullStart = GetFirstHighLowInFilteredS(filteredS(1), glaettung, True)     'Ersten Hochpunkt suchen
        S2NullEnde = GetFirstHighLowInFilteredS(filteredS(1), glaettung, False)     'Ersten Tiefpunkt suchen
        If S2NullStart < S2NullEnde Then        'Hochpunkt kommt zuerst: Nulllinie von erstem Hochpunkt zu letzten Hochpunkt
            S2NullEnde = GetLastHighLowInFilteredS(filteredS(1), glaettung, True)
        Else                                    'Tiefpunkt kommt zuerst: Nulllinie von erstem Tiefpunkt zu letzten Tiefpunkt
            S2NullStart = S2NullEnde
            S2NullEnde = GetLastHighLowInFilteredS(filteredS(1), glaettung, False)
        End If
        S2NullWert = GetNullwertFromFilteredS(filteredS(1), S2NullStart, S2NullEnde)
        'Ermittlung der Nulldurchgaenge
        Dim S2NullIndexes() As Double
        S2NullIndexes = GetNullDurchgaengeInFilteredS(filteredS(1), S2NullStart, S2NullEnde, S2NullWert)    'Indexe sind auf Basis der Messdaten, aber evtl Fliekomma
        'Ermittlung der Parameter der einzelnen Sinusperioden
        Dim S2Periods() As SinusPeriode
        S2Periods = GetSinusPeriods(S2NullIndexes, S2NullWert, 2)
        'Anzeige des Graphen
        Using gr As Graphics = Graphics.FromImage(bi3)
            gr.Clear(Color.Green)
            gr.SmoothingMode = SmoothingMode.AntiAlias
            penline.Color = Color.GreenYellow
            oldx = RANDL
            oldy = CInt(S2Ynull - DataMess(0).Sensor2 * S2YPixProVal)
            For n = 1 To DataMess.Length - 1
                newx = CInt(Math.Round(DataMess(n).Index / DataPos.Length * (P_S2.Width - RANDL - RANDR), 0)) + RANDL 'X-Skalierung orientiert sich an den Indexen in der Pos-Grafik
                newy = CInt(S2Ynull - DataMess(n).Sensor2 * S2YPixProVal)
                gr.DrawLine(penline, oldx, oldy, newx, newy)
                oldx = newx
                oldy = newy
            Next
            'Einzeichnen der gegltteten Kurve, inkl. Nulllinie und Grenzen der Nulllinienfindung
            penline.Color = Color.Blue
            oldx = RANDL
            oldy = CInt(S1Ynull - filteredS(1)(0) * S2YPixProVal)
            For n = 1 To DataMess.Length - 1
                newx = CInt(Math.Round(DataMess(n).Index / DataPos.Length * (P_S2.Width - RANDL - RANDR), 0)) + RANDL   'X-Skalierung orientiert sich an den Indexen in der Pos-Grafik
                newy = CInt(S2Ynull - filteredS(1)(n) * S2YPixProVal)
                gr.DrawLine(penline, oldx, oldy, newx, newy)
                oldx = newx
                oldy = newy
            Next
            oldx = CInt(Math.Round(DataMess(S2NullStart).Index / DataPos.Length * (P_S2.Width - RANDL - RANDR), 0)) + RANDL   'X-Skalierung orientiert sich an den Indexen in der Pos-Grafik
            gr.DrawLine(penline, oldx, 0, oldx, P_S2.Height - 1)
            newx = CInt(Math.Round(DataMess(S2NullEnde).Index / DataPos.Length * (P_S2.Width - RANDL - RANDR), 0)) + RANDL   'X-Skalierung orientiert sich an den Indexen in der Pos-Grafik
            gr.DrawLine(penline, newx, 0, newx, P_S2.Height - 1)
            newy = CInt(S2Ynull - S2NullWert * S2YPixProVal)
            gr.DrawLine(penline, oldx, newy, newx, newy)
            'Einzeichnen der Nulldurchgnge und der finalen Sinuse
            penline.Color = Color.Red
            For n = 0 To S2NullIndexes.Length - 1
                newx = CInt(Math.Round(GetPosIndexFromMessIndex(S2NullIndexes(n)) / DataPos.Length * (P_S2.Width - RANDL - RANDR), 0)) + RANDL   'X-Skalierung orientiert sich an den Indexen in der Pos-Grafik, NullIndexes sind auf Basis der Messdaten
                gr.DrawLine(penline, newx, 0, newx, P_S2.Height - 1)
            Next
            For n = 0 To S2Periods.Length - 1
                oldx = CInt(Math.Round(GetPosIndexFromMessIndex(S2Periods(n).start) / DataPos.Length * (P_S2.Width - RANDL - RANDR), 0)) + RANDL   'X-Skalierung orientiert sich an den Indexen in der Pos-Grafik, StartIndex ist auf Basis der Messdaten
                oldy = CInt(S2Ynull - S2NullWert * S2YPixProVal)
                For m = GetPosIndexFromMessIndex(S2Periods(n).start + 1) To GetPosIndexFromMessIndex(S2Periods(n).ende)
                    newx = CInt(Math.Round(m / DataPos.Length * (P_S2.Width - RANDL - RANDR), 0)) + RANDL                                           'X-Skalierung orientiert sich an den Indexen in der Pos-Grafik
                    newy = CInt(S2Ynull - SinusWert(S2Periods(n).Amplitude, m, GetPosIndexFromMessIndex(S2Periods(n).start), GetPosIndexFromMessIndex(S2Periods(n).ende), S2NullWert) * S2YPixProVal)
                    gr.DrawLine(penline, oldx, oldy, newx, newy)
                    oldx = newx
                    oldy = newy
                Next
            Next
        End Using
        P_S2.Image = bi3


        '****Berechnung und Anzeige des Auswuchtgewichts
        Dim fm, tp, mu, mu1, mu2, musum As Double
        Dim dauer As Integer
        'Anteil des Sensor1 am Auswuchtgewicht (Durchschnitt ber alle Perioden)
        musum = 0
        For n = 0 To S1Periods.Length - 1
            fm = S1Periods(n).Amplitude / KalibS1 / 1000 * 9.81         'Umrechnung der gemessenen Masse in die entsprechende Gewichtskraft in Newton
            dauer = GetPosIndexFromMessIndex(S1Periods(n).ende) - GetPosIndexFromMessIndex(S1Periods(n).start)  'Dauer der Periode in Positionssensor-Indexen
            tp = dauer * 0.001024                   'Periodendauer in Sek, Positionsensor sampelt genau alle 1,024ms (256us*4=1,024ms)
            mu = fm * tp ^ 2 / (4 * RadRadius * Math.PI ^ 2) 'Unwucht-Masse ist mu = fm * tp^2 / (4 * r * pi^2) (Fliehkraft-Formel fm = mu * r * 4 * pi^2 / tp^2) 
            musum = musum + mu
        Next
        mu1 = musum / S1Periods.Length
        L_S1Gramm.Text = (mu1 * 1000).ToString("F1") + " g"
        'Anteil des Sensor2 am Auswuchtgewicht (Durchschnitt ber alle Perioden)
        musum = 0
        For n = 0 To S2Periods.Length - 1
            fm = S2Periods(n).Amplitude / KalibS2 / 1000 * 9.81         'Umrechnung der gemessenen Masse in die entsprechende Gewichtskraft in Newton
            dauer = GetPosIndexFromMessIndex(S2Periods(n).ende) - GetPosIndexFromMessIndex(S2Periods(n).start)  'Dauer der Periode in Positionssensor-Indexen
            tp = dauer * 0.001024                   'Periodendauer in Sek, Positionsensor sampelt genau alle 1,024ms (256us*4=1,024ms)
            mu = fm * tp ^ 2 / (4 * RadRadius * Math.PI ^ 2) 'Unwucht-Masse ist mu = fm * tp^2 / (4 * r * pi^2) (Fliehkraft-Formel fm = mu * r * 4 * pi^2 / tp^2) 
            musum = musum + mu
        Next
        mu2 = musum / S2Periods.Length
        L_S2Gramm.Text = (mu2 * 1000).ToString("F1") + " g"

        '****Berechnung und Anzeige des Winkels zwischen Punkt der Unwucht und Positions-Referenzpunkt am Rad
        'Punkt der Unwucht ist der Punkt, an dem die Unwucht genau auf 12Uhr, also ganz oben steht. Der Sinus ist an dem Punkt minimal, da dort die Fliehkraft der Unwucht der Gewichtskraft des Rads maximal entgegenwirkt
        Dim phsum, ph, ph1, ph2, messindex As Double
        Dim tiefposindex, dist As Integer
        'Winkel, der sich aus Messungen am Sensor 1 ergibt (Durchschnitt ber alle Perioden)
        phsum = 0
        For n = 0 To S1Periods.Length - 1
            messindex = S1Periods(n).start + S1Periods(n).Dauer * 0.75      'Da jede Sinus-Periode bei null und steigender Halbperiode startet, liegt der Tiefpunkt immer bei 3/4 der Periode
            tiefposindex = GetPosIndexFromMessIndex(messindex)              'Entsprechenden Pos-Index dieses Tiefpunkts ermitteln
            m = PosPeaks.Length - 1                                      'Den ersten vor dem gefundenen Tiefpunkt liegenden Peak suchen
            While PosPeaks(m) > tiefposindex
                m = m - 1
            End While
            If m < PosPeaks.Length - 1 Then     'Normalfall: Es gibt noch einen PosPeak nach dem letzten Minimum
                dist = PosPeaks(m + 1) - PosPeaks(m)
            Else                                'Selten aber mglich, Es gibt keinen PosPeak mehr nach dem letzten Minimum - als Periodendauer die der letzten Periode nehmen
                dist = PosPeaks(m) - PosPeaks(m - 1)
            End If
            ph = 360 * (tiefposindex - PosPeaks(m)) / (dist)     'Winkel in Grad berechnen (Position des Tiefpunkts innerhalb der Periode zwischen dem Peak vor und dem Peak nach dem Tiefpunkt)
            phsum = phsum + ph
        Next
        ph1 = phsum / S1Periods.Length
        L_S1Grad.Text = ph1.ToString("F0") + " "
        'Winkel, der sich aus Messungen am Sensor 2 ergibt (Durchschnitt ber alle Perioden)
        phsum = 0
        For n = 0 To S2Periods.Length - 1
            messindex = S2Periods(n).start + S2Periods(n).Dauer * 0.75      'Da jede Sinus-Periode bei null und steigender Halbperiode startet, liegt der Tiefpunkt immer bei 3/4 der Periode
            tiefposindex = GetPosIndexFromMessIndex(messindex)              'Entsprechenden Pos-Index dieses Tiefpunkts ermitteln
            m = PosPeaks.Length - 1                                      'Den ersten vor dem gefundenen Tiefpunkt liegenden Peak suchen
            While PosPeaks(m) > tiefposindex
                m = m - 1
            End While
            If m < PosPeaks.Length - 1 Then     'Normalfall: Es gibt noch einen PosPeak nach dem letzten Minimum
                dist = PosPeaks(m + 1) - PosPeaks(m)
            Else                                'Selten aber mglich, Es gibt keinen PosPeak mehr nach dem letzten Minimum - als Periodendauer die der letzten Periode nehmen
                dist = PosPeaks(m) - PosPeaks(m - 1)
            End If
            ph = 360 * (tiefposindex - PosPeaks(m)) / (PosPeaks(m + 1) - PosPeaks(m))     'Winkel in Grad berechnen (Position des Tiefpunkts innerhalb der Periode zwischen dem Peak vor und dem Peak nach dem Tiefpunkt)
            phsum = phsum + ph
        Next
        ph2 = phsum / S2Periods.Length
        L_S2Grad.Text = ph2.ToString("F0") + " "
        'Die Gesamt-Unwuchtsmasse ist die Amplitude der berlagerung der beiden Sinusfunktionen, je nach Einzel-Amplituden und Einzel-Phasen, da sich das Gewicht des Rads und der Unwucht auf beiden Sensoren verteilt
        mu = Math.Sqrt(mu1 ^ 2 + mu2 ^ 2 + 2 * mu1 * mu2 * Math.Cos(2 * Math.PI * (ph1 - ph2) / 360)) * 1000        'berlagerung der zwei Sinus-Funktionen und Umrechnung von kg in Gramm
        L_Gramm.Text = mu.ToString("F1")
        L_Gramm.ForeColor = Color.Black
        'Die Phase der Ergebnisfunktion der zwei berlagerten Sinus-Funktionen
        Dim tzaehl, tnenn As Double
        tzaehl = mu1 * Math.Sin(2 * Math.PI * ph1 / 360) + mu2 * Math.Sin(2 * Math.PI * ph2 / 360)
        tnenn = mu1 * Math.Cos(2 * Math.PI * ph1 / 360) + mu2 * Math.Cos(2 * Math.PI * ph2 / 360)
        ph = Math.Atan(tzaehl / tnenn) * 360 / (2 * Math.PI)
        'Bei Arctan muss man verschiedene Quadranten unterscheiden, da ein Wert unterschiedliche Ergebnisse haben kann
        If tzaehl > 0 And tnenn > 0 Then        'Gegenkathete und Ankathete sind >0, d.h. Quadrant der Winkel von 0 bis 90: Passt so
            ph = ph
        ElseIf tzaehl < 0 And tnenn > 0 Then    'Gegenkathete<0 und Ankathete >0, d.h. Quadrant der Winkel von 270 bis 360: Arctan wird negativ (-90 bis 0), Ergebnis soll aber positiv sein, deswegen +360
            ph = ph + 360
        Else                                    'Ankathete <0, d.h. Quadrant der Winkel von 90 bis 270*, Atan liefert hier stattdessen Ergebnisse aus den jeweils gegenberliegenden Quadranten: bei 90-180 kommt Ergebnis aus Quadrant 270-360, bei 180-270 kommt Ergebnis aus Quadrant 0-90
            ph = ph + 180
        End If
        L_Grad.Text = ph.ToString("F0")
        L_Grad.ForeColor = Color.Black
        'Dieser Winkel hat ein Problem: Der Abnahme-Punkt fr die Positionssensoren liegt nicht unbedingt genau unten also da wo auch der Unwuchtspunkt erfasst wird
        'Die Kraft-Tief- und Hochpunkte liegen dagegen physikalisch immer genau oben bzw. unten, wenn die Fliehkraft sich maximal zu oder von der Gewichtskraft dazuaddiert oder subtrahiert
        'D.h. es kommt ein schwer ermittelbarer Zusatzwinkel ins Spiel, nmlich der zwischen der Position des Positionssensors und dem untersten Punkt
        'Deswegen gibt es das Ermittlungsverfahren in den Anwisungen von Zusand 7-9. Siehe dazu auch "Ermittlungen.ods"
    End Sub

    Private Sub xDisplayAndProcessData()
        'Alternative zu der groen Auswertung - zur reinen Datenaufnahme: Zeichnet die Graphen der 3 Messreihen (Position, Sens1, Sens2) ohne weiteres und ohne Berechnungen
        Dim n As Integer
        Dim penline As Pen
        Dim oldx, oldy, newx, newy As Integer

        '***** Anzeige der Positionssensor-Daten ********
        Dim PosMax, PosMin As Byte
        Dim PosYPixProVal As Double      'Skalierung in Pixel/Value
        Dim PosYRange, PosYnull As Double
        Dim bi As New Bitmap(P_Pos.Width, P_Pos.Height)
        'Ermittlung von Min/Max-Werten und Skalierungen
        PosMin = DataPos(0)
        PosMax = DataPos(0)
        For n = 1 To DataPos.Length - 1
            If DataPos(n) < PosMin Then PosMin = DataPos(n)
            If DataPos(n) > PosMax Then PosMax = DataPos(n)
        Next
        If PosMax = PosMin Then
            PosYRange = 1
        Else
            PosYRange = PosMax - PosMin
        End If
        PosYPixProVal = (P_Pos.Height - RANDO - RANDU) / PosYRange          'Y-Skalierung Pixel/Value
        PosYnull = PosYPixProVal * PosMax + RANDO                           'Ymax liegt an oberem Anzeigerand, 0-Linie entsprechend unterhalb
        'Anzeige des Graphen
        Using gr As Graphics = Graphics.FromImage(bi)
            gr.Clear(Color.Green)
            gr.SmoothingMode = SmoothingMode.AntiAlias
            penline = New Pen(Color.GreenYellow, 1)
            oldx = RANDL
            oldy = CInt(PosYnull - DataPos(0) * PosYPixProVal)
            For n = 1 To DataPos.Length - 1
                newx = CInt(Math.Round(n / DataPos.Length * (P_Pos.Width - RANDL - RANDR), 0)) + RANDL
                newy = CInt(PosYnull - DataPos(n) * PosYPixProVal)
                gr.DrawLine(penline, oldx, oldy, newx, newy)
                oldx = newx
                oldy = newy
            Next
        End Using
        P_Pos.Image = bi


        '***** Anzeige der Sensor1-Daten ********
        Dim S1Max, S1Min As Integer
        Dim S1YPixProVal As Double      'Skalierung in Pixel/Value
        Dim S1YRange, S1Ynull As Double
        Dim bi2 As New Bitmap(P_S1.Width, P_S1.Height)
        'Ermittlung von Min/Max-Werten und Skalierungen
        S1Min = DataMess(0).Sensor1
        S1Max = DataMess(0).Sensor1
        For n = 1 To DataMess.Length - 1
            If DataMess(n).Sensor1 < S1Min Then S1Min = DataMess(n).Sensor1
            If DataMess(n).Sensor1 > S1Max Then S1Max = DataMess(n).Sensor1
        Next
        If S1Max = S1Min Then
            S1YRange = 1
        Else
            S1YRange = S1Max - S1Min
        End If
        S1YPixProVal = (P_S1.Height - RANDO - RANDU) / S1YRange          'Y-Skalierung Pixel/Value
        S1Ynull = S1YPixProVal * S1Max + RANDO                           'Ymax liegt an oberem Anzeigerand, 0-Linie entsprechend unterhalb
        'Anzeige des Graphen
        Using gr As Graphics = Graphics.FromImage(bi2)
            gr.Clear(Color.Green)
            gr.SmoothingMode = SmoothingMode.AntiAlias
            penline.Color = Color.GreenYellow
            oldx = RANDL
            oldy = CInt(S1Ynull - DataMess(0).Sensor1 * S1YPixProVal)
            For n = 1 To DataMess.Length - 1
                newx = CInt(Math.Round(DataMess(n).Index / DataPos.Length * (P_S1.Width - RANDL - RANDR), 0)) + RANDL   'X-Skalierung orientiert sich an den Indexen in der Pos-Grafik
                newy = CInt(S1Ynull - DataMess(n).Sensor1 * S1YPixProVal)
                gr.DrawLine(penline, oldx, oldy, newx, newy)
                oldx = newx
                oldy = newy
            Next
        End Using
        P_S1.Image = bi2


        '***** Anzeige der Sensor2-Daten ********
        Dim S2Max, S2Min As Integer
        Dim S2YPixProVal As Double      'Skalierung in Pixel/Value
        Dim S2YRange, S2Ynull As Double
        Dim bi3 As New Bitmap(P_S2.Width, P_S2.Height)
        'Ermittlung von Min/Max-Werten und Skalierungen
        S2Min = DataMess(0).Sensor2
        S2Max = DataMess(0).Sensor2
        For n = 1 To DataMess.Length - 1
            If DataMess(n).Sensor2 < S2Min Then S2Min = DataMess(n).Sensor2
            If DataMess(n).Sensor2 > S2Max Then S2Max = DataMess(n).Sensor2
        Next
        If S2Max = S2Min Then
            S2YRange = 1
        Else
            S2YRange = S2Max - S2Min
        End If
        S2YPixProVal = (P_S2.Height - RANDO - RANDU) / S2YRange          'Y-Skalierung Pixel/Value
        S2Ynull = S2YPixProVal * S2Max + RANDO                           'Ymax liegt an oberem Anzeigerand, 0-Linie entsprechend unterhalb
        'Anzeige des Graphen
        Using gr As Graphics = Graphics.FromImage(bi3)
            gr.Clear(Color.Green)
            gr.SmoothingMode = SmoothingMode.AntiAlias
            penline.Color = Color.GreenYellow
            oldx = RANDL
            oldy = CInt(S2Ynull - DataMess(0).Sensor2 * S2YPixProVal)
            For n = 1 To DataMess.Length - 1
                newx = CInt(Math.Round(DataMess(n).Index / DataPos.Length * (P_S2.Width - RANDL - RANDR), 0)) + RANDL 'X-Skalierung orientiert sich an den Indexen in der Pos-Grafik
                newy = CInt(S2Ynull - DataMess(n).Sensor2 * S2YPixProVal)
                gr.DrawLine(penline, oldx, oldy, newx, newy)
                oldx = newx
                oldy = newy
            Next
        End Using
        P_S2.Image = bi3
    End Sub

    Private Function GetPosPeakIndexes(ByVal min As Byte, ByVal max As Byte) As Integer()
        'Ermittelt die zeitlichen Indices aller Peaks in dem DataPos-Array
        Dim n, rise, fall As Integer
        Dim indexes(0) As Integer
        Dim schwellwert As Byte
        Dim found As Boolean

        schwellwert = Math.Round((max - min) * 0.75 + min)    'Der Schwellwert zur Flankenerkennung soll bei 50% zwischen Nulllinie und Max-Wert sein (Nullinie wird angenommen als genau bei 50% zw. Min und Max)
        n = 0
        While DataPos(n) > schwellwert And n < DataPos.Length         'Wenn Datensatz mit Werten > Schwellwert beginnt (Mesststart innherhalb eines Peaks), diese berspringen - bis Werte unter den Schwellwert kommen
            n = n + 1
        End While
        n = n + 3       '3 Werte berspringen um Rausch-Artifakte zu unterdrcken, so schnell (3 ms) sollten eh keine zwei Pos-Peaks aufeinander folgen
        While n < DataPos.Length
            found = False
            While n < DataPos.Length And found = False       'Steigende Flanke finden
                If DataPos(n) >= schwellwert Then               'Gefunden
                    found = True
                    rise = n
                End If
                n = n + 1
            End While
            If found = True Then
                n = n + 3       '3 Werte berspringen um Rausch-Artifakte zu unterdrcken, so schnell (3 ms) sollte ein Peak eh nicht vorbei sein
                found = False
                While n < DataPos.Length And found = False       'Fallende Flanke finden
                    If DataPos(n) <= schwellwert Then               'Gefunden
                        found = True
                        fall = n
                    End If
                    n = n + 1
                End While
            End If
            If found = True Then        'Found ist hier nur dann true, wenn sowohl rise als auch fall gefunden wurden
                indexes(indexes.Length - 1) = Math.Round((fall - rise) / 2 + rise)  'Peak-Punkt ist genau in der Mitte zw. steigender und fallender Flanke
                ReDim Preserve indexes(indexes.Length)
                n = n + 3       '3 Werte berspringen um Rausch-Artifakte zu unterdrcken, so schnell (3 ms) sollten eh keine zwei Flanken aufeinander folgen
            End If
        End While
        ReDim Preserve indexes(indexes.Length - 2)
        Return indexes
    End Function

    Private Function GetFilteredCurve(ByVal kan As Byte, ByVal anz As Byte) As Integer()
        'Ermittelt die stark geglaettete Version der Messkurve
        'Filterung ist Moving Avereage ber 2*anz+1 Werte (von x-anz bis x+anz). Das glttet sehr stark, aber ohne eine Phasenverschiebung einzubringen
        'Die Dmpfung ist zwar sehr stark, das ist aber irrelevant. Gesucht sind lediglich die Nullline des Sinus und die entsprechenden 0-Durchgnge der einzelnen Perioden
        'Diese werden durch das Verfahren nicht verndert
        Dim n, m As Integer
        Dim result(DataMess.Length - 1) As Integer

        For n = 0 To anz - 1            'Die ersten anz Werte werden so bernommen, ohne Glttung
            If kan = 1 Then result(n) = DataMess(n).Sensor1
            If kan = 2 Then result(n) = DataMess(n).Sensor2
        Next
        For n = anz To DataMess.Length - anz - 1    'Die Werte ab anz werden geglttet 
            result(n) = 0
            For m = n - anz To n + anz
                If kan = 1 Then result(n) = result(n) + DataMess(m).Sensor1
                If kan = 2 Then result(n) = result(n) + DataMess(m).Sensor2
            Next
            result(n) = Math.Round(result(n) / (2 * anz + 1))
        Next
        For n = DataMess.Length - anz To DataMess.Length - 1    'Die letzten anz Werte werden wieder ohne Glttung bernommen
            If kan = 1 Then result(n) = DataMess(n).Sensor1
            If kan = 2 Then result(n) = DataMess(n).Sensor2
        Next
        Return result
    End Function

    Private Function GetFirstHighLowInFilteredS(ByVal filteredS() As Integer, ByVal glwert As Integer, ByVal high As Boolean) As Integer
        'Findet den ersten Hoch oder Tiefpunkt in der gegltteten Kurve eines Kanals
        'Weitere Filterung ist ntig, da Kurve noch ein Restrauschen hat. Einzelne Hoch/Tiefpunkt zu suchen wrde auf lokale Spikes reinfallen
        'Hochpunkt ist der Punkt bei dem der Durchschnitt der umliegenden Werte hher ist als der Durchschnitt der Werte davor und der Werte danach
        'Tiefpunkt ist entsprechend der Punkt bei dem der Durchschnitt der umliegenden Werte kleiner ist als der Durchschnitt der Werte davor und der Werte danach
        'Index des Ergebnisses ist auf Basis der Messdaten-Indices, d.h. muss evtl noch auf Basis der Positionsindices umgerechnet werden
        Dim n, m As Integer
        Dim halbweite, weite As Byte
        Dim vorher, nachher, mitte As Double
        Dim found As Boolean

        halbweite = 3                       'Soviele Werte vorher+nachher zusammen mit dem aktuellen Wert mitteln
        weite = 2 * halbweite + 1

        found = False
        n = glwert + weite               'Erst bei ab hier anfangen zu suchen, davor ist Kurve ungeglttet
        While n < filteredS.Length - 1 - glwert - weite And found = False
            mitte = 0                                   'Der Duchschnitt der Werte um den aktuellen Punkt
            For m = n - halbweite To n + halbweite
                mitte = mitte + filteredS(m)
            Next
            mitte = mitte / weite
            vorher = 0                                  'Der Duchschnitt der Werte vor dem aktuellen Punkt
            For m = n - weite To n - 1
                vorher = vorher + filteredS(m)
            Next
            vorher = vorher / weite
            nachher = 0                                 'Der Duchschnitt der Werte nach dem aktuellen Punkt
            For m = n + 1 To n + weite
                nachher = nachher + filteredS(m)
            Next
            nachher = nachher / weite
            If high = True Then                     'Es soll ein Hochpunkt gesucht werden
                If mitte > vorher And mitte > nachher Then
                    found = True
                Else
                    n = n + 1
                End If
            Else                                    'Es soll ein Tiefpunkt gesucht werden
                If mitte < vorher And mitte < nachher Then
                    found = True
                Else
                    n = n + 1
                End If
            End If
        End While
        If found = True Then
            Return n
        Else
            Return 0
        End If
    End Function

    Private Function GetLastHighLowInFilteredS(ByVal filteredS() As Integer, ByVal glwert As Integer, ByVal high As Boolean) As Integer
        'Findet den letzten Hoch oder Tiefpunkt in der gegltteten Kurve eines Kanals
        'Weitere Filterung ist ntig, da Kurve noch ein Restrauschen hat. Einzelnen Hoch/Tiefpunkt zu suchen wrde auf lokale Spikes reinfallen
        'Hochpunkt ist der Punkt bei dem der Durchschnitt der umliegenden Werte hher ist als der Durchschnitt der Werte davor und der Werte danach
        'Tiefpunkt ist entsprechend der Punkt bei dem der Durchschnitt der umliegenden Werte kleiner ist als der Durchschnitt der Werte davor und der Werte danach
        'Index des Ergebnisses ist auf Basis der Messdaten-Indices, d.h. muss evtl noch auf Basis der Positionsindices umgerechnet werden
        Dim n, m As Integer
        Dim halbweite, weite As Byte
        Dim vorher, nachher, mitte As Double
        Dim found As Boolean

        halbweite = 3                       'Soviele Werte vorher+nachher zusammen mit dem aktuellen Wert mitteln
        weite = 2 * halbweite + 1

        found = False
        n = filteredS.Length - 1 - glwert - weite               'Erst bei ab hier anfangen zu suchen, davor ist Kurve ungeglttet
        While n > glwert + weite And found = False
            mitte = 0                                   'Der Duchschnitt der Werte um den aktuellen Punkt
            For m = n - halbweite To n + halbweite
                mitte = mitte + filteredS(m)
            Next
            mitte = mitte / weite
            vorher = 0                                  'Der Duchschnitt der Werte vor dem aktuellen Punkt
            For m = n - weite To n - 1
                vorher = vorher + filteredS(m)
            Next
            vorher = vorher / weite
            nachher = 0                                 'Der Duchschnitt der Werte nach dem aktuellen Punkt
            For m = n + 1 To n + weite
                nachher = nachher + filteredS(m)
            Next
            nachher = nachher / weite
            If high = True Then                 'Es soll ein Hochpunkt gesucht werden
                If mitte > vorher And mitte > nachher Then
                    found = True
                Else
                    n = n - 1
                End If
            Else                                'Es soll ein Tiefpunkt gesucht werden
                If mitte < vorher And mitte < nachher Then
                    found = True
                Else
                    n = n - 1
                End If
            End If

        End While
        If found = True Then
            Return n
        Else
            Return 0
        End If
    End Function

    Private Function GetNullwertFromFilteredS(ByVal filteredS() As Integer, ByVal start As Integer, ByVal ende As Integer) As Double
        'Ermittelt den Nullwert der gegltteten Kurve
        'Da Start und Stop jeweils entweder beide Hoch oder beide Tiefpunkte sind, der Bereich als vollstndige Perioden umfasst, gengt eine reine Mittlung ber alle Werte
        Dim erg As Double
        Dim n As Integer

        erg = 0
        For n = start To ende
            erg = erg + filteredS(n)
        Next
        erg = erg / (ende - start + 1)
        Return erg
    End Function

    Private Function GetNullDurchgaengeInFilteredS(ByVal filteredS() As Integer, ByVal start As Integer, ByVal ende As Integer, ByVal Nullwert As Integer) As Double()
        'Ermittelt alle positiven Nulldurchgnge der Filtered-Kurve innerhalb der gesetzten Grenzen
        'Weitere Filterung ist ntig, da die Kurve noch ein Restrauschen hat. Einzelnen Nulldurchgang zu suchen wrde auf lokale Spikes reinfallen
        'Nullduchgangspunkt ist der Punkt in der Mitte der Zone, die durch folgende Bedingungen begrenzt wird:
        'Start der Zone: Letzter Index bei dem der Durchschnitt der Punkte vorher und der Durchschnitt der Punkte nachher < Nullwert ist
        'Ende der Zone: Erster Index bei dem der Durchschnitt der Punkte vorher und der Durchschnitt der Punkte nachher > Nullwert ist
        'Index der Ergebnisse ist auf Basis der Messdatendaten-Indices, ist aber evtl eine Fliekommazahl! Fr Verwendung als Positionsindex muss diese noch umgerechnet werden (GetPosIndexFromMessIndex) 
        Dim n, m, szone, ezone As Integer
        Dim weite As Byte
        Dim vorher, nachher, mitte As Double
        Dim sfound As Boolean
        Dim ndindexes(0) As Double

        weite = 4                       'Soviele Werte vorher bzw nachher (inkl aktuellem Wert) mitteln

        sfound = False
        n = start + weite
        While n < ende - weite
            vorher = 0                                  'Der Duchschnitt der Werte vor dem aktuellen Punkt
            For m = n - weite + 1 To n
                vorher = vorher + filteredS(m)
            Next
            vorher = vorher / weite
            nachher = 0                                 'Der Duchschnitt der Werte nach dem aktuellen Punkt
            For m = n To n + weite - 1
                nachher = nachher + filteredS(m)
            Next
            nachher = nachher / weite
            If sfound = False And vorher < Nullwert And nachher > Nullwert Then     'Zonenstart gefunden
                szone = n - 1               'Start der Zone ist letzter Index, bei dem nachher noch nicht grer als Nullwert war, deswegen -1
                sfound = True
            End If
            If sfound = True And vorher > Nullwert And nachher > Nullwert Then      'Zonenende gefunden
                ezone = n                   'Ende der Zone ist erster Index, bei dem vorher ger als der Nullwert ist, deswegen direkt n nehmen
                sfound = False
                'Berechnung des Nulldurchgangsindex, indem einfach die Mitte der Zone genommen wird. Stimmt das wirklich? Was wenn Zonenstart deutlich nher am Nullwert liegt als Zonenende?
                'mitte = szone + (ezone - szone) / 2         'Nulldurchgang ist genau in der Mitte der Zone 
                'Berechnung des Nulldurchgangsindex, indem die gewichtete Mitte der Zone genommen wird
                vorher = vorher - Nullwert      'Der zuletzt ermittelte vorher-Wert ist noch immer der erste Durchschnittswert, der groeer als der Nullwert ist. Abstand zur Nullinie errechnen
                nachher = 0                     'Der letzte nachher-Wert, der kleiner als der Nullwert war, ist bereits berschrieben. Muss neu ermittelt werden
                For m = szone To szone + weite - 1
                    nachher = nachher + filteredS(m)
                Next
                nachher = nachher / weite
                nachher = Nullwert - nachher    'Abstand zur Nullinie errechnen, nachher-Wert am Start der Zone ist dadurch definiert, dass er der letzte nachher-Wert ist, der kleiner als der Nullwert ist
                mitte = szone + (ezone - szone) * nachher / (nachher + vorher)      'Nulldurchgang in der gewichteten Mitte der Zone
                ndindexes(ndindexes.Length - 1) = mitte
                ReDim Preserve ndindexes(ndindexes.Length)
            End If
            n = n + 1
        End While
        ReDim Preserve ndindexes(ndindexes.Length - 2)
        Return ndindexes
    End Function

    Private Function GetSinusPeriods(ByVal NullIndexes() As Double, ByVal Nullwert As Integer, ByVal kan As Byte) As SinusPeriode()
        'Ermittelt die Basisdaten (Amplitude und Periodendauer) der einzelnen Sinusperioden in der Rohwert-Kurve von Kanal kan
        'Periodendauer ist recht einfach: nur die Differenz zwischen den einzelnen Nulldurchgangs-Indices
        'Die Amplituden werden schrittweise fr jede Vollperiode angenhert
        'Dazu wird die Amplitude solange verndert, bis die Abweichung von mathematischem Sinus und Rohwerten der Periode (Quadrat-Summe von Messwert minus Annherungswert) ein Minimum erreicht.
        'Das bietet einige Vorteile:
        '- Perfekte Glttung ohne eine Dmpfung
        '- Perfekte Glttung ohne eine Phasenverschiebung
        '- Perfekte Interpolation durch Ermittlung einer mathematischen Formel
        'Die Sinus-Perioden werden einzeln ermittelt, da sich im Laufe einer Messung die Periodenlnge vergrern und die Amplitude verkleinern kann (Rad dreht mit der Zeit immer langsamer, Fliehkraft durch Unwucht wird entsprechend geringer)
        Dim periods(0) As SinusPeriode
        Dim amplitude As Integer
        Dim dauer As Double
        Dim stepsize(2), StartIndex, EndIndex, messwert, mathwert As Integer
        Dim QuadSumAlt, QuadSumNeu, Quad As Long     'Long ist fr die Berechnung ntig wenn 24Bit-Werte quadriert und aufsummiert werden
        Dim n, messindex As Integer

        stepsize(0) = 2000      '2000 entspricht bei 24-Bit-Werten ca. einem Gewicht von 10g
        stepsize(1) = 500       '500 entspricht bei 24-Bit-Werten ca. einem Gewicht von 4g
        stepsize(2) = 200       '200 entspricht bei 24-Bit-Werten ca. einem Gewicht von 1g
        'stepsize(0) = 9         '9 entspricht bei 16-Bit-Werten ca. einem Gewicht von 10g
        'stepsize(1) = 4         '4 entspricht bei 16-Bit-Werten ca. einem Gewicht von 4g
        'stepsize(2) = 1         '1 entspricht bei 16-Bit-Werten ca. einem Gewicht von 1g
        For n = 0 To NullIndexes.Length - 2                 'Fr jede einzelne Periode (Zonen zwischen den Nulldurchgang-Indexes)
            dauer = NullIndexes(n + 1) - NullIndexes(n)
            QuadSumAlt = 0
            QuadSumNeu = 0
            amplitude = 0
            StartIndex = Math.Ceiling(NullIndexes(n))   'Start beim ersten Messwert innerhalb der Periode
            EndIndex = Math.Floor(NullIndexes(n + 1))   'Ende beim letzten Messwert innerhalb der Periode
            For messindex = StartIndex To EndIndex    'Quadrat-Summe von (Messwert-Mathwert) bei amplitude=0 ermitteln
                If kan = 1 Then messwert = DataMess(messindex).Sensor1
                If kan = 2 Then messwert = DataMess(messindex).Sensor2
                mathwert = SinusWert(amplitude, messindex, NullIndexes(n), NullIndexes(n + 1), Nullwert)
                Quad = (messwert - mathwert) ^ 2
                QuadSumNeu = QuadSumNeu + Quad
            Next
            QuadSumAlt = QuadSumNeu + 1             '+1 damit Neu<Alt und folgende while-Schleife mindestens einmal durchlaufen wird
            While QuadSumNeu < QuadSumAlt                   'Amplitude solange um Stepsize(0) erhhen, bis die daraus resultierende Quadratsumme grer als die zuletzt ermittelte wird (Minimum wurde durchschritten)
                QuadSumAlt = QuadSumNeu
                QuadSumNeu = 0
                amplitude = amplitude + stepsize(0)
                For messindex = StartIndex To EndIndex    'Quadrat-Summe von (Messwert-Mathwert) bei aktueller amplitude ermitteln
                    If kan = 1 Then messwert = DataMess(messindex).Sensor1
                    If kan = 2 Then messwert = DataMess(messindex).Sensor2
                    mathwert = SinusWert(amplitude, messindex, NullIndexes(n), NullIndexes(n + 1), Nullwert)
                    Quad = (messwert - mathwert) ^ 2
                    QuadSumNeu = QuadSumNeu + Quad
                Next
            End While
            QuadSumAlt = QuadSumNeu + 1             '+1 damit Neu<Alt und folgende while-Schleife mindestens einmal durchlaufen wird
            While QuadSumNeu < QuadSumAlt                   'Amplitude solange um Stepsize(1) reduzieren, bis die daraus resultierende Quadratsumme grer als die zuletzt ermittelte wird (Minimum wurde durchschritten)
                QuadSumAlt = QuadSumNeu
                QuadSumNeu = 0
                amplitude = amplitude - stepsize(1)
                For messindex = StartIndex To EndIndex    'Quadrat-Summe von (Messwert-Mathwert) bei aktueller amplitude ermitteln
                    If kan = 1 Then messwert = DataMess(messindex).Sensor1
                    If kan = 2 Then messwert = DataMess(messindex).Sensor2
                    mathwert = SinusWert(amplitude, messindex, NullIndexes(n), NullIndexes(n + 1), Nullwert)
                    Quad = (messwert - mathwert) ^ 2
                    QuadSumNeu = QuadSumNeu + Quad
                Next
            End While
            QuadSumAlt = QuadSumNeu + 1             '+1 damit Neu<Alt und folgende while-Schleife mindestens einmal durchlaufen wird
            While QuadSumNeu < QuadSumAlt                   'Amplitude solange um Stepsize(2) erhhen, bis die daraus resultierende Quadratsumme grer als die zuletzt ermittelte wird (Minimum wurde durchschritten)
                QuadSumAlt = QuadSumNeu
                QuadSumNeu = 0
                amplitude = amplitude + stepsize(2)
                For messindex = StartIndex To EndIndex    'Quadrat-Summe von (Messwert-Mathwert) bei aktueller amplitude ermitteln
                    If kan = 1 Then messwert = DataMess(messindex).Sensor1
                    If kan = 2 Then messwert = DataMess(messindex).Sensor2
                    mathwert = SinusWert(amplitude, messindex, NullIndexes(n), NullIndexes(n + 1), Nullwert)
                    Quad = (messwert - mathwert) ^ 2
                    QuadSumNeu = QuadSumNeu + Quad
                Next
            End While
            amplitude = amplitude - stepsize(2)             'Gefunden wurde der erste Werte, der hher als das Minimum ist. D.h. Minimum ist ein Wert davor
            periods(periods.Length - 1).Dauer = dauer
            periods(periods.Length - 1).Amplitude = amplitude
            periods(periods.Length - 1).start = NullIndexes(n)
            periods(periods.Length - 1).ende = NullIndexes(n + 1)
            ReDim Preserve periods(periods.Length)
        Next
        ReDim Preserve periods(periods.Length - 2)
        Return periods
    End Function

    Private Function SinusWert(ByVal amplitude As Integer, ByVal index As Integer, ByVal start As Integer, ByVal ende As Integer, ByVal nullwert As Integer) As Integer
        'Die Ermittung eines Sinuswerts mit den entsprechenden Parametern
        Return nullwert + Math.Round(amplitude * Math.Sin(2 * Math.PI * (index - start) / ((ende - start))))
    End Function

    Private Function GetPosIndexFromMessIndex(ByVal MessIndex As Double) As Integer
        'Ermittelt aus einem Index auf Basis der Messdaten (kann Fliekomma sein) den zugehrigen Index auf Basis der Positionsdaten
        'Normalerweise (bei Integer Messdaten-Index) ist das einfach der Index, der in der Structure des Messdatums festgehalten ist - MessDatum(MessIndex).Index
        'Bei einem Fliekomma-Index wird hier ein entsprechender Zwischen-Index in der Basis der Positionsdaten ermittelt
        '(Positionsdaten werden mit hherer Sample-Rate aufgenommen als Messdaten, daher stimmen die Indexe nicht berein)
        Dim sindex, eindex, pindex As Integer
        Dim deci As Double

        sindex = DataMess(Math.Floor(MessIndex)).Index          'Randindexe auf Basis der Positionsdaten ermitteln (MessIndex kann Fliekomma sein)
        eindex = DataMess(Math.Ceiling(MessIndex)).Index        'MessIndex ist z.B. 75,3 (auf Messdaten-Basis)- Auf Positionsdaten-Basis entspricht 75 dann z.B. Index 1000 und 76 entspricht Index 1010
        deci = MessIndex - Math.Floor(MessIndex)                'deci ist dann 0,3
        pindex = sindex + Math.Round((eindex - sindex) * deci)  'bei Messdatenindex 75,3 ist Positionsdaten-Index dann 1003
        Return pindex
    End Function

End Class